#pragma once

#include <yamail/data/serialization/json_writer.h>
#include <ymod_webserver/server.h>
#include <mailbox_oper/errors.h>
#include <mail/mops/include/internal/server/task_traits.h>
#include <mailbox_oper/context.h>
#include <butil/network/real_ip_from_headers.h>

namespace mbox_oper {

using OptString = boost::optional<std::string>;

struct Response {
    std::string result;
    std::string error;
    Response(const std::string& result_, const std::string& error_)
        : result(result_), error(error_) {
    }
};

struct OperationResponse {
    std::string result;
    std::string error;
    std::string taskType;
    OptString taskGroupId;

    OperationResponse(const std::string& result_, const std::string& error_,
                      const TaskTraits& task)
        : result(result_), error(error_),
          taskType(toString(task.type)),
          taskGroupId() {
        if (task.groupId) {
            taskGroupId = boost::uuids::to_string(*task.groupId);
        }
    }
};

class ResponseFactory {
public:
    static OperationResponse createSuccessResponse(const TaskTraits& data) {
        return OperationResponse("ok", "", data);
    }

    static Response createInternalErrorResponse(const std::string& error) {
        return Response("internal error", error);
    }

    static Response createInvalidRequestResponse(const std::string& error) {
        return Response("invalid request", error);
    }

    static Response createForbiddenResponse(const std::string& error) {
        return Response("forbidden", error);
    }

    static Response createMethodNotAllowedResponse(const ymod_webserver::methods::http_method expectedMethod) {
        using namespace ymod_webserver;
        const auto methodName = [](const methods::http_method method) {
            switch (method) {
                case methods::mth_post: return "POST";
                case methods::mth_get: return "GET";
                default: return "Unknown method";
            }
        };

        std::ostringstream os;
        os << "you should make " << methodName(expectedMethod) << " request";
        return Response("method not allowed", os.str());
    }

    static Response createNotFoundResponse(const std::string& error) {
        return Response("not found", error);
    }
};

struct TaskContext final : Context {
    TaskContext(const std::string& uniqId, const std::string& requestId)
        : Context(uniqId, requestId) {}

    const std::string& get_name() const override {
        static const std::string name("mops");
        return name;
    }
};

inline const std::string& getHeader(const ymod_webserver::request& request, const std::string& name) {
    static const std::string emptyHeader = "";
    const auto iter = request.headers.find(boost::algorithm::to_lower_copy(name));
    return iter == request.headers.end() ? emptyHeader : iter->second;
}

inline OptString getOptionalHeader(const ymod_webserver::request& request, const std::string& name) {
    const auto iter = request.headers.find(boost::algorithm::to_lower_copy(name));
    return iter == request.headers.end() ? boost::none : boost::make_optional(iter->second);
}

inline const std::string& getRequestId(const ymod_webserver::request& request) {
    return getHeader(request, "X-Request-Id");
}

inline std::string getRealIp(const ymod_webserver::request& request) {
    return real_ip::getFromHeaders(getOptionalHeader(request, "X-Forwarded-For"),
                                   getOptionalHeader(request, "X-Real-Ip"),
                                   request.context->remote_address);
}

class RequestContext {
public:
    RequestContext(ymod_webserver::response_ptr response)
        : request_(response->request())
        , response_(response)
        , context_(boost::make_shared<TaskContext>(request_->ctx()->uniq_id(), getRequestId(*request_)))
    {}

    void responseSuccess(const TaskTraits& data) {
        const auto answer = ResponseFactory::createSuccessResponse(data);
        response(buildAnswer(answer), ymod_webserver::codes::ok);
    }

    void responseSuccess() {
        response("{}", ymod_webserver::codes::ok);
    }

    void responseSuccess(const std::string& data) {
        response(data, ymod_webserver::codes::ok);
    }

    void responseInternalError(const std::string& error) {
        const auto answer = ResponseFactory::createInternalErrorResponse(error);
        response(buildAnswer(answer), ymod_webserver::codes::internal_server_error);
    }

    void responseInvalidRequest(const std::string& error) {
        const auto answer = ResponseFactory::createInvalidRequestResponse(error);
        response(buildAnswer(answer), ymod_webserver::codes::bad_request);
    }

    void responseForbidden(const std::string& error) {
        const auto answer = ResponseFactory::createForbiddenResponse(error);
        response(buildAnswer(answer), ymod_webserver::codes::forbidden);
    }

    void responseMethodNotAllowed(const ymod_webserver::methods::http_method expectedMethod) {
        const auto answer = ResponseFactory::createMethodNotAllowedResponse(expectedMethod);
        response(buildAnswer(answer), ymod_webserver::codes::method_not_allowed);
    }

    void responseNotFound(const std::string& error) {
        const auto answer = ResponseFactory::createNotFoundResponse(error);
        response(buildAnswer(answer), ymod_webserver::codes::not_found);
    }

    void rejectByTvmTicket() {
        response("", ymod_webserver::codes::unauthorized);
    }

    std::string getArg(const std::string& name) const {
        const auto& params = request_->url.params;
        const auto iter = params.find(name);
        return iter == params.end() ? "" : iter->second;
    }

    OptString getArgOptional(const std::string& name) const {
        const auto& params = request_->url.params;
        const auto iter = params.find(name);
        return iter == params.end() ? OptString() : iter->second;
    }

    OptBool getArgOptionalBool(const std::string& name) const {
        const OptString param = getArgOptional(name);
        if (!param) {
            return OptBool();
        }
        const std::string val = boost::algorithm::to_lower_copy(*param);
        if (val == "1" || val == "yes") {
            return true;
        } else if (val == "0" || val == "no") {
            return false;
        } else {
            throw ParamsException("invalid value for bool argument " + name);
        }
    }

    std::string getArgOrThrow(const std::string& name) const {
        const auto& params = request_->url.params;
        const auto iter = params.find(name);
        if (iter == params.end() || iter->second == "") {
            throw ParamsException("parameter not found: " + name);
        }
        return iter->second;
    }

    std::string getArgOrValue(const std::string& name, const std::string& value) const {
        const auto& params = request_->url.params;
        const auto iter = params.find(name);
        if (iter == params.end() || iter->second == "") {
            return value;
        }
        return iter->second;
    }

    template <typename T>
    T getArg(const std::string& name, const T& def = T()) const {
        std::string arg = getArg(name);
        if (arg.empty()) {
            return def;
        }
        try {
            return boost::lexical_cast<T>(arg);
        } catch (const boost::bad_lexical_cast& e) {
            throw ParamsException("invalid argument for " + name + ", bad_lexical_cast: " + e.what());
        }
    }

    std::string body() const {
        const auto& body = request_->raw_body;
        return std::string(body.begin(), body.end());
    }

    const std::string& getHeader(const std::string& name) const {
        return mbox_oper::getHeader(*request_, name);
    }

    OptString getOptionalHeader(const std::string& name) const {
        return mbox_oper::getOptionalHeader(*request_, name);
    }

    const std::string& requestId() const {
        return mbox_oper::getRequestId(*request_);
    }

    const ymod_webserver::request& request() const noexcept {
        return *request_;
    }

    ContextPtr context() const {
        return context_;
    }

    std::string realIp() const {
        return mbox_oper::getRealIp(*request_);
    }

private:
    void response(const std::string& str, ymod_webserver::codes::code code) {
        response_->set_code(code);
        response_->set_content_type("application", "json");
        response_->result_body(str);
    }

    template <typename Resp>
    std::string buildAnswer(const Resp& answer) {
        return yamail::data::serialization::toJson(answer).str();
    }

    ymod_webserver::request_ptr request_;
    ymod_webserver::response_ptr response_;
    boost::shared_ptr<TaskContext> context_;
};

}

BOOST_FUSION_ADAPT_STRUCT(mbox_oper::Response,
    (std::string, result)
    (std::string, error)
)

BOOST_FUSION_ADAPT_STRUCT(mbox_oper::OperationResponse,
    (std::string, result)
    (std::string, error)
    (std::string, taskType)
    (mbox_oper::OptString, taskGroupId)
)

