#pragma once

#include <yamail/data/deserialization/json_reader.h>
#include <mailbox_oper/logger.h>
#include <mailbox_oper/yield_context.h>
#include <mailbox_oper/context.h>
#include <mail/http_getter/client/include/module.h>


namespace mbox_oper::blackbox {

using namespace std::literals;

namespace reflection {
using Aliases = std::map<std::string, std::string>;
} // namespace reflection

} // namespace mbox_oper::blackbox

BOOST_FUSION_DEFINE_STRUCT((mbox_oper)(blackbox)(reflection), StringValue,
    (std::optional<std::string>, value)
)

BOOST_FUSION_DEFINE_STRUCT((mbox_oper)(blackbox)(reflection), Exception,
    (std::string, value)
    (int, id)
)

BOOST_FUSION_DEFINE_STRUCT((mbox_oper)(blackbox)(reflection), User,
    (mbox_oper::blackbox::reflection::StringValue, uid)
    (std::optional<mbox_oper::blackbox::reflection::Aliases>, aliases)
    (std::optional<mbox_oper::blackbox::reflection::StringValue>, karma)
    (std::optional<mbox_oper::blackbox::reflection::StringValue>, karma_status)
    (std::optional<std::string>, login)
)

BOOST_FUSION_DEFINE_STRUCT((mbox_oper)(blackbox)(reflection), Response,
    (std::optional<std::vector<mbox_oper::blackbox::reflection::User>>, users)
    (std::optional<std::string>, error)
    (std::optional<mbox_oper::blackbox::reflection::Exception>, exception)
)

namespace mbox_oper::blackbox::error {
enum Codes {
    ok,
    badResponse,
    userDoesNotExist,
    nonRetryableStatus,
    retriesExceeded,
};
} // namespace mbox_oper::blackbox::error

namespace boost::system {

template <>
struct is_error_code_enum<mbox_oper::blackbox::error::Codes> : std::true_type {};

} // namespace boost::system

namespace mbox_oper::blackbox {

namespace error {

class Category : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "mbox_oper::blackbox::error::Category";
    }

    std::string message(int v) const override {
        switch(Codes(v)) {
            case ok:
                return "no error";
            case badResponse:
                return "bad blackbox response";
            case userDoesNotExist:
                return "specified user does not exist";
            case nonRetryableStatus:
                return "HTTP response status is not retryable";
            case retriesExceeded:
                return "retries possabilities exceeded";
        }
        return "unknown error";
    }
};

inline const boost::system::error_category& category() {
    static Category instance;
    return instance;
}

inline boost::system::error_code make_error_code(Codes e) {
    return {static_cast<int>(e), category()};
}

} // namespace error

using boost::system::system_error;

struct User {
    bool isMailish = false;
    std::string uid;
    std::string login;
    std::string karma;
    std::string karmaStatus;
};

inline unsigned status(const ymod_httpclient::response& r) {
    return r.status;
}

inline bool retryable(const ymod_httpclient::response& r) {
    return status(r) / 100 == 5;
}

inline bool succeeded(const ymod_httpclient::response& r) {
    return status(r) == 200;
}

inline reflection::Response parse(const ymod_httpclient::response& r) {
    return yamail::data::deserialization::fromJson<reflection::Response>(r.body);
}

inline bool failed(const reflection::Response& r) {
    return bool(r.error);
}

inline auto& errorMessage(const reflection::Response& r) {
    if (!r.error) {
        throw system_error{error::badResponse, "bad result, no error object found"};
    }
    return *r.error;
}

inline const reflection::User& user(const reflection::Response& r) {
    if (!r.users) {
        throw system_error{error::badResponse, "bad result, no users object found"};
    } else if (r.users->size() != 1) {
        throw system_error{error::badResponse, "bad users count "s + std::to_string(r.users->size())};
    }
    return r.users->front();
}

inline bool exists(const reflection::User& user) {
    return bool(user.uid.value);
}

inline auto& aliases(const reflection::User& user) {
    if (!user.aliases) {
        throw system_error{error::badResponse, "no aliases provided"};
    }
    return *user.aliases;
}

inline bool isMailish(const reflection::User& user) {
    return aliases(user).count("12");
}

inline User fromReflection(const reflection::User& user) {
    User res;
    res.isMailish = isMailish(user);
    res.uid = user.uid.value.value_or("");
    res.login = user.login.value_or("");
    if (user.karma) {
        res.karma = user.karma->value.value_or("");
    }
    if (user.karma_status) {
        res.karmaStatus = user.karma_status->value.value_or("");
    }

    return res;
}

template <typename Logger>
struct UserInfoOp {
    http_getter::TypedClientPtr httpClient;
    http_getter::TypedEndpoint endpoint;
    Logger logger_;


    UserInfoOp(http_getter::TypedClientPtr httpClient, http_getter::TypedEndpoint endpoint, Logger logger)
        : httpClient(std::move(httpClient))
        , endpoint(std::move(endpoint))
        , logger_(std::move(logger))
    { }

    User perform(const std::string& uid, const std::string& userIp, YieldCtx yield) const {
        auto logger = ::logdog::bind(logger_, log::service="blackbox");
        using namespace http_getter;
        auto req = httpClient->toGET(endpoint)
            .getArgs(
                "method"_arg = "userinfo"s,
                "uid"_arg = uid,
                "userip"_arg = userIp,
                "format"_arg = "json"s,
                "aliases"_arg = "12"s
            );
        
        auto err = error::ok;
        std::optional<User> userInfo;
        http_getter::Handler handler;
        handler.error = [&](auto ec) {
            if (!ec) return;
            LOGDOG_(logger, notice, log::error_code=ec);
        };
        handler.success = [&](auto response) {
            err = error::ok;
            if (!succeeded(response)) {
                err = error::nonRetryableStatus;
                return http_getter::Result::fail;
            }

            auto res = parse(response);
            if (failed(res)) {
                LOGDOG_(logger, notice, log::message=errorMessage(res));
                return http_getter::Result::retry;
            }
            if (res.exception) {
                LOGDOG_(logger, notice, log::message=res.exception->value);
                return http_getter::Result::retry;
            }

            auto usr = user(res);
            if (!exists(usr)) {
                LOGDOG_(logger, error, log::message="user does not exist");
                err = error::userDoesNotExist;
                return http_getter::Result::fail;
            }

            userInfo = fromReflection(usr);
            if (userInfo->isMailish) {
                LOGDOG_(logger, notice, log::message = "user is mailish");
            }
            return http_getter::Result::success;
        };

        httpClient->req(std::move(req))->call("blackbox", std::move(handler), io_result::make_yield_context(yield));
        
        if (userInfo) {
            return *userInfo;
        }
        if (err != error::ok) {
            throw boost::system::system_error(err);
        }

        throw boost::system::system_error(error::retriesExceeded);
    }
};

template <typename Logger>
inline User userInfo(const http_getter::TypedClientPtr& httpClient, const http_getter::TypedEndpoint& endpoint, const std::string& uid,
        const std::string& userIp, Logger logger, YieldCtx yield) {
    auto op = mbox_oper::blackbox::UserInfoOp{httpClient, endpoint, logger};
    return op.perform(uid, userIp, yield);
}

}
