#ifndef MOPER_ERRORS_H_2014_1027
#define MOPER_ERRORS_H_2014_1027

#include <macs/errors.h>

#include <stdexcept>

namespace mbox_oper {

struct ParamsException: public std::runtime_error {
    ParamsException(const std::string& msg)
        : std::runtime_error(msg)
    {}
};

struct MailboxOperException: public std::runtime_error {
    MailboxOperException(const std::string& msg)
        : std::runtime_error(msg) {
    }
    MailboxOperException(const std::string& msg, const std::string& paramsDescr)
        : std::runtime_error(msg + " " + paramsDescr) {
    }
};

class ConfigurationException: public std::runtime_error {
public:
    ConfigurationException(const std::string& msg)
        : std::runtime_error(msg) {
    }
};

struct RetriableException: public MailboxOperException {
    RetriableException(const std::string& msg)
        : MailboxOperException(msg)
    {}
};

struct ReadOnlyMetabaseException: public MailboxOperException {
    ReadOnlyMetabaseException(const std::string& msg)
        : MailboxOperException(msg)
    {}
};

struct DatbaseLockFailed: public RetriableException {
    DatbaseLockFailed(const std::string& msg)
        : RetriableException(msg)
    {}
};

namespace error {

using error_code = macs::error_code;
using error_category = error_code::error_category;

enum Errors {
    ok,
    folderDoesntExists,
    tabDoesntEsist
};

} // namespace error
} // namespace mbox_oper

namespace boost {
namespace system {

template <>
struct is_error_code_enum<mbox_oper::error::Errors> {
    static const bool value = true;
};

} // namespace system
} // namespace boost

namespace mbox_oper {
namespace error {

class Category : public error_category {
public:
    const char* name() const noexcept override {
        return "mbox_oper::error::Category";
    }

    std::string message(int v) const override {
        switch(Errors(v)) {
            case ok :
                return "no error";
            case folderDoesntExists :
                return "folder doesn't exists";
            case tabDoesntEsist :
                return "tab doesn't exist";
        }
        return "unknown error";
    }
};

inline const error_category& getCategory() {
    static Category instance;
    return instance;
}

inline error_code::base_type make_error_code(Errors e) {
    return error_code::base_type(static_cast<int>(e), getCategory());
}

} // namespace error
} // namespace mbox_oper

#endif // MOPER_ERRORS_H_2014_1027
