#pragma once

#include <mailbox_oper/yield_context.h>
#include <mailbox_oper/config.h>
#include <mailbox_oper/logger.h>
#include <macs/folder.h>
#include <macs/label.h>
#include <macs/envelope.h>
#include <macs_pg/service/service.h>
#include <macs/data/folder_sort_options.h>


namespace mbox_oper {

struct MailboxModifier {
    using Fid = macs::Fid;
    using Lid = macs::Lid;
    using Folder = macs::Folder;
    using Label = macs::Label;
    using Envelope = macs::Envelope;
    using Mids = std::list<macs::Mid>;
    using error_code = macs::error_code;
    using Tab = macs::Tab;

    virtual ~MailboxModifier() = default;

    virtual void moveMessages(Fid, std::optional<Tab::Type>, Mids, YieldCtx)const = 0;
    virtual void updateStatus(Mids, Envelope::Status, YieldCtx)const = 0;
    virtual void remove(Mids, YieldCtx)const = 0;
    virtual void changeEnvelopesLabels(Mids, std::list<Label> add, std::list<Label> del, YieldCtx)const = 0;

    virtual Folder createFolder(const std::string& name, const Fid& parent,
        Folder::Symbol, YieldCtx)const = 0;
    virtual Folder getOrCreateFolder(const std::string& name, const Fid& parent,
        Folder::Symbol, YieldCtx)const = 0;
    virtual Folder getOrCreateFolderBySymbolWithRandomizedName(const std::string& name, const Fid& parent,
        Folder::Symbol, bool forceRandomize, YieldCtx)const = 0;
    virtual void updateFolder(Folder, YieldCtx)const = 0;
    virtual void eraseCascade(Folder, YieldCtx)const = 0;

    virtual void setPop3(const std::vector<Fid>& fids, YieldCtx)const = 0;

    virtual void setSortOptions(const Fid& fid, const Fid& prevFid, YieldCtx)const = 0;
    virtual void setFolderSymbol(const Folder&, Folder::Symbol, YieldCtx)const = 0;
    virtual void resetFolderSymbol(const Folder&, YieldCtx)const = 0;

    virtual Label createLabel(Label::Symbol symbol, YieldCtx)const = 0;
    virtual void deleteLabel(Lid, YieldCtx)const = 0;
    virtual Label getOrCreateLabel(Label::Symbol symbol, YieldCtx)const = 0;
    virtual Label createLabel(const std::string& name, const std::string& color,
        Label::Type type, YieldCtx)const = 0;
    virtual Label getOrCreateLabel(const std::string& name, const std::string& color,
        Label::Type type, YieldCtx)const = 0;
    virtual void updateLabel(Label, YieldCtx)const = 0;
};

struct MailboxModifierMacsImpl : MailboxModifier {
    macs::ServicePtr mailbox_;

    MailboxModifierMacsImpl(macs::ServicePtr mailbox) : mailbox_(mailbox) {}

    static auto wrap(YieldCtx yield) { return make_yield_context(yield); }

    void moveMessages(Fid fid, std::optional<Tab::Type> tabType,
                      Mids mids, YieldCtx yield) const override {
        mailbox_->envelopes().moveMessages(fid, tabType, mids, wrap(yield));
    }

    void updateStatus(Mids mids, Envelope::Status status, YieldCtx yield) const override {
        mailbox_->envelopes().updateStatus(mids, status, wrap(yield));
    }

    void remove(Mids mids, YieldCtx yield) const override {
        mailbox_->envelopes().remove(mids, wrap(yield));
    }

    void changeEnvelopesLabels(Mids mids, std::list<Label> add, std::list<Label> del, YieldCtx yield) const override {
        mailbox_->envelopes().changeEnvelopesLabels(mids, add, del, wrap(yield));
    }

    Folder createFolder(const std::string& name, const Fid& parent,
            Folder::Symbol symbol, YieldCtx yield) const override {
        return mailbox_->folders().createFolder(name, parent, symbol, wrap(yield));
    }

    Folder getOrCreateFolderBySymbolWithRandomizedName(const std::string& name, const Fid& parent,
            Folder::Symbol symbol, bool forceRandomize, YieldCtx yield) const override {
        return macs::getOrCreateFolderBySymbolWithRandomizedName(mailbox_->folders(), name, parent, symbol, forceRandomize, wrap(yield));
    }

    Folder getOrCreateFolder(const std::string& name, const Fid& parent,
            Folder::Symbol symbol, YieldCtx yield) const override {
        return mailbox_->folders().getOrCreateFolder(name, parent, symbol, wrap(yield));
    }

    void updateFolder(Folder folder, YieldCtx yield) const override {
        mailbox_->folders().updateFolder(folder, wrap(yield));
    }

    void eraseCascade(Folder folder, YieldCtx yield) const override {
        mailbox_->folders().eraseCascade(folder, wrap(yield));
    }

    void setPop3(const std::vector<Fid>& fids, YieldCtx yield) const override {
        mailbox_->folders().setPop3(fids, wrap(yield));
    }

    void setSortOptions(const Fid& fid, const Fid& prevFid, YieldCtx yield) const override {
        macs::SortOptions sortOptions(mailbox_->folders());
        sortOptions.move(fid, prevFid, wrap(yield));
    }

    void setFolderSymbol(const Folder& folder, Folder::Symbol symbol, YieldCtx yield) const override {
        mailbox_->folders().setSymbol(folder, symbol, wrap(yield));
    }

    void resetFolderSymbol(const Folder& folder, YieldCtx yield) const override {
        mailbox_->folders().resetSymbol(folder, wrap(yield));
    }

    Label createLabel(Label::Symbol sym, YieldCtx yield) const override {
        return mailbox_->labels().createLabel(sym, wrap(yield));
    }

    void deleteLabel(Lid lid, YieldCtx yield) const override {
        mailbox_->labels().deleteLabel(lid, wrap(yield));
    }

    Label getOrCreateLabel(Label::Symbol sym, YieldCtx yield) const override {
        return mailbox_->labels().getOrCreateLabel(sym, wrap(yield));
    }

    Label createLabel(const std::string& name, const std::string& color, Label::Type type, YieldCtx yield) const override {
        return mailbox_->labels().createLabelWithType(name, color, type.title(), wrap(yield));
    }

    Label getOrCreateLabel(const std::string& name, const std::string& color,
            Label::Type type, YieldCtx yield) const override {
        return mailbox_->labels().getOrCreateLabel(name, color, type, wrap(yield));
    }

    void updateLabel(Label label, YieldCtx yield) const override {
        mailbox_->labels().updateLabel(label, wrap(yield));
    }
};

} // namespace mbox_oper

