#pragma once

#include <functional>
#include <macs/envelope.h>
#include <macs/thread_mailbox_list.h>
#include <user_journal/journal.h>
#include <mailbox_oper/operations.h>
#include <mailbox_oper/mailbox_meta.h>
#include <mailbox_oper/mailbox_modifier.h>

namespace mbox_oper {

class MailboxOper {
public:
    MailboxOper(
        macs::ServicePtr metadata,
        std::shared_ptr<MailboxModifier> mailboxModifier,
        ContextPtr context,
        const user_journal::Journal& journal,
        const macs::deprecated::TabsMap& tabsMap);

    template <typename Params>
    void execute(const Operation<Params>& operation, YieldCtx yieldCtx) {
        try {
            exec(operation.mids, operation.params, yieldCtx);
        } catch (const macs::MailboxLockTimeoutException& e) {
            throw RetriableException(std::string("lock mailbox timeout, can retry: ") + e.what());
        } catch (const macs::ReadOnlyException& e) {
            throw ReadOnlyMetabaseException(std::string("db-readonly exception: ") + e.what());
        }
    }

    macs::Folder createFolder(const std::string& name, const macs::Fid parent,
        macs::Folder::Symbol, YieldCtx);
    macs::Folder getOrCreateFolder(const std::string& name, const macs::Fid parent,
        macs::Folder::Symbol, YieldCtx);
    macs::Folder getOrCreateFolderBySymbolWithRandomizedName(const std::string& name, const macs::Fid& parent,
        macs::Folder::Symbol, bool forceRandomize, YieldCtx);
    void updateFolder(const macs::Fid& fid, const boost::optional<std::string>& name,
        const boost::optional<macs::Fid>& parentFid, YieldCtx);
    void setPop3(const std::vector<macs::Fid>& fids, YieldCtx);
    void setSortOptions(const macs::Fid& fid, const macs::Fid& prevFid, YieldCtx);
    void setFolderSymbol(const macs::Fid& fid, const std::string& symbol, YieldCtx);
    macs::Label createLabel(const boost::optional<std::string>& symbol, YieldCtx);
    macs::Label getOrCreateLabel(const boost::optional<std::string>& symbol, YieldCtx);
    macs::Label createLabel(const std::string& name, const std::string& color,
        const std::string& typeName, YieldCtx);
    macs::Label getOrCreateLabel(const std::string& name, const std::string& color,
        const std::string& typeName, YieldCtx);
    void updateLabel(const macs::Lid& lid, const boost::optional<std::string>& color,
        const boost::optional<std::string>& name, YieldCtx);
private:
    macs::Service& metadata() const noexcept {
        return *metadata_;
    }
public:
    const MailboxMeta& meta() const noexcept {
        return meta_;
    }

    const macs::deprecated::TabsMap& tabsMap() const {
        return tabsMap_;
    }

private:
    const MailboxModifier& modifyMailbox() const {
        return *mailboxModifier_;
    }

    void exec(const Mids& mids, const MarkParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const PurgeParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const SpamParams& params, YieldCtx yieldCtx);
    void exec(const macs::ThreadMailboxItems& messages, const SpamParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const UnspamParams& params, YieldCtx yieldCtx);
    void exec(const macs::ThreadMailboxItems& messages, const UnspamParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const MoveParams& params, YieldCtx yieldCtx);
    void exec(const macs::ThreadMailboxItems& messages, const MoveParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const ComplexMoveParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const TrashParams& params, YieldCtx yieldCtx);
    void exec(const macs::ThreadMailboxItems& items, const TrashParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const RemoveParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const LabelParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const UnlabelParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const DeleteLabelParams& params, YieldCtx yieldCtx);

    void exec(const Mids& mids, const DeleteFolderParams& params, YieldCtx yieldCtx);

    MidsPair splitRemoveMessages(const Mids& mids, YieldCtx yieldCtx);
    MidsPair splitRemoveMessages(const Mids& mids, const Fid& fid, YieldCtx yieldCtx);
    ComplexMoveMailboxSet splitComplexMoveMessages(const Mids& mids,
            const std::string& destFid, YieldCtx yieldCtx);


    using ThreadMailboxItemsPair = std::pair<macs::ThreadMailboxItems, macs::ThreadMailboxItems>;

    ThreadMailboxItemsPair splitMessagesByFid(macs::ThreadMailboxItems messages, const macs::Fid& fid);
    ThreadMailboxItemsPair splitMessagesByFids(macs::ThreadMailboxItems messages, const std::vector<macs::Fid>& fids);

    void trash(const macs::ThreadMailboxItems& messages, YieldCtx yieldCtx);


    macs::ThreadMailboxItems getMailboxItems(const Mids& mids, YieldCtx yieldCtx);
    macs::ThreadMailboxItems getMailboxItems(const Mids& mids, const macs::Lids& lidsToCheck, YieldCtx yieldCtx);
    macs::ThreadMailboxItems getUnreadMailboxItems(const macs::ThreadMailboxItems& messages) const;

    void spamMoveAndMark(const Mids& mids, const bool shouldMove, const std::string& spamFid,
                         YieldCtx yieldCtx);

    std::optional<macs::Tab::Type> resolveDestTab(const OptString& destTab) const;

    void logAbuse(const macs::ThreadMailboxItems& messages, const std::string& type, bool hidden) const;

    macs::ServicePtr metadata_;
    MailboxMetaImpl meta_;
    ContextLogger logger_;
    ContextPtr context_;
    user_journal::Journal journal_;
    macs::deprecated::TabsMap tabsMap_;
    std::shared_ptr<MailboxModifier> mailboxModifier_;

    static constexpr auto emptyTab = std::nullopt; //< for move from tab operations
};

} // namespace mailbox_oper

