#pragma once

#include <string>
#include <sstream>
#include <list>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/optional.hpp>
#include <mailbox_oper/errors.h>
#include <macs/types.h>
#include <macs/envelope.h>
#include <macs/folder.h>
#include <macs/tabs_map.h>
#include <macs/thread_mailbox_list.h>
#include <chrono>
#include <yamail/data/reflection/access.h>
#include <mailbox_oper/yield_context.h>
#include <mail/mops/include/common/params.h>
#include <compare>

namespace mbox_oper {

using OptBool = boost::optional<bool>;
using OptString = boost::optional<std::string>;

using Mid = std::string;
using Tid = std::string;
using Mids = std::list<Mid>;
using Tids = std::list<Tid>;
using MidsPair = std::pair<Mids, Mids>;

class Fid {
    std::string value;
public:
    Fid() = default;
    explicit Fid(std::string value)
        : value(std::move(value))
    {}

    operator const std::string&() const {
        return value;
    }

    bool operator== (const Fid& that) const noexcept {
        return value == that.value;
    }

    bool empty() const noexcept {
        return value.empty();
    }
};

inline std::ostream& operator<< (std::ostream& stream, const Fid& fid) {
    stream << static_cast<const std::string&>(fid);
    return stream;
}

using OptFid = boost::optional<Fid>;

class Lid {
    std::string value;
public:
    Lid() = default;
    explicit Lid(std::string value)
        : value(std::move(value))
    {}

    operator const std::string&() const {
        return value;
    }

    bool operator== (const Lid& that) const noexcept {
        return value == that.value;
    }

    bool empty() const noexcept {
        return value.empty();
    }
};

inline std::ostream& operator<< (std::ostream& stream, const Lid& lid) {
    stream << static_cast<const std::string&>(lid);
    return stream;
}

using Lids = std::vector<Lid>;
using Fids = std::vector<Fid>;

inline std::string joinIds(const std::list<std::string> ids) {
    return boost::algorithm::join(ids, ",");
}

using OptStatus = boost::optional<macs::Envelope::Status>;

struct ComplexMoveMailboxSet {
    macs::ThreadMailboxItems spamMailboxItems;
    macs::ThreadMailboxItems trashMailboxItems;
    macs::ThreadMailboxItems moveMailboxItems;
    macs::ThreadMailboxItems unspamMailboxItems;
};

inline macs::Envelope::Status statusFromString(const std::string& status) {
    if (status == "read") {
        return macs::Envelope::Status_read;
    } else if (status == "not_read") {
        return macs::Envelope::Status_unread;
    } else if (status == "replied") {
        return macs::Envelope::Status_replied;
    } else if (status == "forwarded") {
        return macs::Envelope::Status_forwarded;
    } else {
        throw ParamsException("status not supported: " + status);
    }
}

inline std::string statusToString(const macs::Envelope::Status status) {
    switch (status) {
    case macs::Envelope::Status_read:
        return "read";
    case macs::Envelope::Status_unread:
        return "not_read";
    case macs::Envelope::Status_replied:
        return "replied";
    case macs::Envelope::Status_forwarded:
        return "forwarded";
    default:
        throw std::runtime_error("status enum not supported: " + std::to_string(status));
    }
}

inline std::string statusToString(const OptStatus& status) {
    if (status) {
        return statusToString(status.get());
    } else {
        return "";
    }
}

#define GETTER(type, field) \
    type field##_; \
public: \
    std::conditional_t<std::is_integral<type>::value, type, const type&> \
    field() const noexcept { \
        return field##_; \
    } \
private:

struct MidsSourceData {
    using Days = macs::Days;

    Mids mids;
    Tids tids;
    std::string fid;
    std::string lid;
    Days age{ 0 };
    std::string subject;
    std::string from;
    std::string tabName;

    std::string fromMid;
    std::string limit;

    bool operator== (const MidsSourceData& other) const noexcept {
        return mids == other.mids
            && tids == other.tids
            && fid == other.fid
            && lid == other.lid
            && age == other.age
            && subject == other.subject
            && from == other.from
            && tabName == other.tabName
            && fromMid == other.fromMid
            && limit == other.limit
            ;
    }

    std::string toString() const {
        std::ostringstream ss;
        ss << "mids=" << joinIds(mids)
           << "tids=" << joinIds(tids)
           << "fid="  << fid
           << "lid="  << lid
           << "age="  << age.count()
           << "subject=" << subject
           << "from="    << from
           << "tabName=" << tabName
           << "fromMid=" << fromMid
           << "limit=" << limit
           ;
        return ss.str();
    }
};

class MailboxMeta;

using ExcludeFilter = std::function<bool(const macs::Folder&)>;
using ExcludeFilters = std::vector<ExcludeFilter>;
using SkipFolders = std::vector<macs::Folder::Symbol>;

class CascadeOptions {
    GETTER(bool, enable)
    GETTER(ExcludeFilters, excludeFilters)
public:
    explicit CascadeOptions(const bool enable, ExcludeFilters excludeFilters)
        : enable_(enable),
          excludeFilters_(std::move(excludeFilters))
    {}

    CascadeOptions()
        : CascadeOptions(false, {})
    {}

    explicit CascadeOptions(const bool enable)
        : CascadeOptions(enable, {})
    {}

    CascadeOptions(ExcludeFilters excludeFilters)
        : CascadeOptions(true, std::move(excludeFilters))
    {}
};

class ResolveOptions {
    GETTER(SkipFolders, skipFolders)
    GETTER(OptStatus, excludeStatus)
    GETTER(CascadeOptions, cascadeOptions)
    GETTER(bool, syncByHdrMessageId)
public:
    explicit ResolveOptions(SkipFolders skipFolders = SkipFolders(),
                            const OptStatus& excludeStatus = boost::none,
                            CascadeOptions cascade = CascadeOptions(),
                            const bool syncByHdrMessageId = false)
        : skipFolders_(std::move(skipFolders)),
          excludeStatus_(excludeStatus),
          cascadeOptions_(std::move(cascade)),
          syncByHdrMessageId_(syncByHdrMessageId)
    {}
};

inline std::ostream& operator<< (std::ostream& stream, const ResolveOptions& options) {
    std::vector<std::string> skipFoldersNames;
    skipFoldersNames.reserve(options.skipFolders().size());
    boost::transform(options.skipFolders(), std::back_inserter(skipFoldersNames),
                    [](const auto& symbol) { return symbol.title(); });

    stream << "skipFolders=";
    boost::copy(skipFoldersNames, std::ostream_iterator<std::string>(stream, ","));
    stream << " excludeStatus=" << statusToString(options.excludeStatus())
           << " cascade=[enable=" << options.cascadeOptions().enable()
           << ", excludeFiltersCount=" << options.cascadeOptions().excludeFilters().size()
           << "] syncByHdrMessageId=" << options.syncByHdrMessageId();
    return stream;
}

using ValidationError = std::string;
using OptValidationError = boost::optional<ValidationError>;

class MidsSource;
using MidsSourcePtr = std::shared_ptr<MidsSource>;

class MidsSource {
public:
    virtual ~MidsSource() = default;

    virtual OptValidationError validate() const = 0;

    virtual Mids resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                         YieldCtx yieldCtx) const = 0;

    virtual size_t getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                                     YieldCtx yieldCtx) const = 0;

    virtual MidsSourceData getData() const = 0;

    virtual bool isNeedAsyncResolve(const ResolveOptions& options) const;

    virtual bool isNeedPaginate() const;

    virtual MidsSourcePtr paginate(const std::optional<Mid>& fromMid, std::size_t limit) const = 0;
};

class DirectMidsSource : public MidsSource {
    Mids mids;
public:
    explicit DirectMidsSource(Mids mids)
        : mids(std::move(mids))
    {}

    OptValidationError validate() const override;

    Mids resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const override;

    size_t getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                             YieldCtx yieldCtx) const override;

    MidsSourceData getData() const override;

    bool isNeedAsyncResolve(const ResolveOptions& options) const override;

    MidsSourcePtr paginate(const std::optional<Mid>& fromMid, std::size_t limit) const override;

private:
    bool isNeedSyncByHdrMessageId(const ResolveOptions& options) const;
};

class TidsSource : public MidsSource {
    Tids tids;
public:
    explicit TidsSource(Tids tids)
        : tids(std::move(tids))
    {}

    OptValidationError validate() const override;

    Mids resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const override;

    size_t getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                             YieldCtx yieldCtx) const override;

    MidsSourceData getData() const override;

    MidsSourcePtr paginate(const std::optional<Mid>& fromMid, std::size_t limit) const override;

private:
    bool isNeedSyncByHdrMessageId(const ResolveOptions& options) const;
};

class MidsWithTidsSource : public MidsSource {
    Mids mids;
    Tids tids;
public:
    explicit MidsWithTidsSource(Mids mids, Tids tids)
        : mids(std::move(mids)),
          tids(std::move(tids))
    {}

    OptValidationError validate() const override;

    Mids resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const override;

    size_t getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                             YieldCtx yieldCtx) const override;

    MidsSourceData getData() const override;

    MidsSourcePtr paginate(const std::optional<Mid>& fromMid, std::size_t limit) const override;
};

struct FidFilter {
    using Subject = std::string;
    using From = std::string;

    using OptDays = boost::optional<MidsSourceData::Days>;
    using OptSubject = boost::optional<Subject>;
    using OptFrom = boost::optional<From>;

    FidFilter()
        : FidFilter(boost::none, boost::none, boost::none)
    {}

    explicit FidFilter(const unsigned age, Subject subject, From from)
        : FidFilter(age == 0u       ? boost::none : OptDays(age),
                    subject.empty() ? boost::none : OptSubject(subject),
                    from.empty()    ? boost::none : OptFrom(from))
    {}

    explicit FidFilter(OptDays age, OptSubject subject, OptFrom from)
        : age_(std::move(age)),
          subject_(std::move(subject)),
          from_(std::move(from)) {
        if (age_ && age_.get().count() == 0u) {
            throw ParamsException("age needs to be greater than 0");
        }

        if (subject_ && subject_.get().empty()) {
            throw ParamsException("subject needs to be non empty");
        }

        if (from_ && from_.get().empty()) {
            throw ParamsException("from needs to be non empty");
        }
    }

    bool hasConditions() const noexcept {
        return age_ || subject_ || from_;
    }

    GETTER(OptDays, age)
    GETTER(OptSubject, subject)
    GETTER(OptFrom, from)
};

class FidSource : public MidsSource {
    Fid fid;
    FidFilter filter;
public:
    explicit FidSource(Fid fid, FidFilter filter = FidFilter())
        : fid(std::move(fid)),
          filter(std::move(filter))
    {}

    OptValidationError validate() const override;

    Mids resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const override;

    size_t getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                             YieldCtx yieldCtx) const override;

    MidsSourceData getData() const override;

    const Fid& getFid() const {
        return fid;
    }

    const FidFilter& getFilter() const {
        return filter;
    }

    bool isNeedPaginate() const override;

    MidsSourcePtr paginate(const std::optional<Mid>& fromMid, std::size_t limit) const override;
};

class PagedFidSource : public FidSource {
    std::optional<Mid> fromMid;
    std::size_t limit;
public:
    PagedFidSource(FidSource source, std::optional<Mid> fromMid, std::size_t limit)
        : FidSource(std::move(source)),
          fromMid(std::move(fromMid)),
          limit(limit)
    {}

    OptValidationError validate() const override;

    Mids resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const override;

    MidsSourceData getData() const override;
};

class LidSource : public MidsSource {
    Lid lid;
public:
    explicit LidSource(Lid lid)
        : lid(std::move(lid))
    {}

    OptValidationError validate() const override;

    Mids resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const override;

    size_t getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                             YieldCtx yieldCtx) const override;

    MidsSourceData getData() const override;

    MidsSourcePtr paginate(const std::optional<Mid>& fromMid, std::size_t limit) const override;
};

class TabSource: public MidsSource {
    std::string tabName;
public:
    explicit TabSource(std::string tabName)
        : tabName(std::move(tabName)) {}

    OptValidationError validate() const override;

    Mids resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const override;

    size_t getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                             YieldCtx yieldCtx) const override;

    MidsSourceData getData() const override;

    MidsSourcePtr paginate(const std::optional<Mid>& fromMid, std::size_t limit) const override;

private:
    macs::Tab::Type tabType() const {
        return macs::Tab::Type::fromString(tabName, std::nothrow);
    }
};

MidsSourcePtr toMidsSource(const MidsSourceData& data);

class BaseParams {
public:
    virtual ~BaseParams() = default;

    virtual ResolveOptions resolveOptions(const MailboxMeta&, const MailboxOperParams&, YieldCtx) const {
        return ResolveOptions();
    }

    virtual bool breakOnEmptyMids() const {
        return true;
    }

    auto operator<=>(const BaseParams&) const = default;
};

class SpamParams : public BaseParams {
    GETTER(bool, shouldMove)
    GETTER(OptBool, withSent)

    YREFLECTION_GRANT_ACCESS
    SpamParams() = default;
public:
    explicit SpamParams(const bool shouldMove,
                        const OptBool& withSent)
        : BaseParams(),
          shouldMove_(shouldMove),
          withSent_(withSent)
    {}

    ResolveOptions resolveOptions(const MailboxMeta&, const MailboxOperParams&, YieldCtx) const override {
        if (withSent_.get_value_or(true)) {
            return ResolveOptions();
        } else {
            return ResolveOptions({ macs::Folder::Symbol::sent });
        }
    }

    auto operator<=>(const SpamParams&) const = default;
};

class UnspamParams : public BaseParams {
    GETTER(Fid, destFid)
    GETTER(bool, shouldMove)
    GETTER(OptString, destTab)

    YREFLECTION_GRANT_ACCESS
    UnspamParams() = default;
public:
    explicit UnspamParams(Fid destFid,
                          const bool shouldMove,
                          OptString destTab = boost::none)
        : BaseParams(),
          destFid_(std::move(destFid)),
          shouldMove_(shouldMove),
          destTab_(std::move(destTab))
    {}

    auto operator<=>(const UnspamParams&) const = default;
};

class PurgeParams : public BaseParams {
public:
    bool operator== (const PurgeParams&) const noexcept {
        return true;
    }
};

class RemoveParams : public BaseParams {
    GETTER(bool, trashOnly)
    GETTER(OptBool, withSent)
    GETTER(OptFid, fid)

    YREFLECTION_GRANT_ACCESS
    RemoveParams() = default;
public:
    explicit RemoveParams(const bool trashOnly, const OptBool& withSent,
                          OptFid fid)
        : BaseParams(),
          trashOnly_(trashOnly),
          withSent_(withSent),
          fid_(std::move(fid))
    {}

    ResolveOptions resolveOptions(const MailboxMeta&, const MailboxOperParams&, YieldCtx) const override {
        if (withSent_.get_value_or(true)) {
            return ResolveOptions();
        } else {
            return ResolveOptions({ macs::Folder::Symbol::sent });
        }
    }

    bool operator== (const RemoveParams& that) const noexcept {
        return trashOnly_ == that.trashOnly_
            && withSent_ == that.withSent_
            && fid_ == that.fid_;
    }
};

class TrashParams : public BaseParams {
public:
    bool operator== (const TrashParams&) const noexcept {
        return true;
    }
};

class MoveParams : public BaseParams {
    GETTER(Fid, destFid)
    GETTER(OptString, destTab)

    YREFLECTION_GRANT_ACCESS
    MoveParams() = default;
public:
    explicit MoveParams(Fid destFid, OptString destTab = boost::none)
        : BaseParams(),
          destFid_(std::move(destFid)),
          destTab_(std::move(destTab)) {
        if (destFid_.empty()) {
            throw ParamsException("dest_fid must not be empty");
        }
    }

    bool operator== (const MoveParams& that) const noexcept {
        return destFid_ == that.destFid_
            && destTab_ == that.destTab_;
    }
};

class ComplexMoveParams : public BaseParams {
    GETTER(Fid, destFid)
    GETTER(OptBool, withSent)
    GETTER(OptString, destTab)

    YREFLECTION_GRANT_ACCESS
    ComplexMoveParams() = default;
public:
    explicit ComplexMoveParams(Fid destFid, const OptBool& withSent, OptString destTab = boost::none)
        : BaseParams(),
          destFid_(std::move(destFid)),
          withSent_(withSent),
          destTab_(std::move(destTab)) {
        if (destFid_.empty()) {
            throw ParamsException("dest_fid must not be empty");
        }
    }

    ResolveOptions resolveOptions(const MailboxMeta&, const MailboxOperParams&, YieldCtx yieldCtx) const override;

    bool operator== (const ComplexMoveParams& that) const noexcept {
        return destFid_ == that.destFid_
            && withSent_ == that.withSent_
            && destTab_ == that.destTab_;
    }
};

class MarkParams : public BaseParams {
    GETTER(macs::Envelope::Status, status)

    YREFLECTION_GRANT_ACCESS
    MarkParams() = default;
public:
    explicit MarkParams(const macs::Envelope::Status status)
        : BaseParams(),
          status_(status)
    {}

    ResolveOptions resolveOptions(const MailboxMeta&, const MailboxOperParams&, YieldCtx) const override;

    bool operator== (const MarkParams& that) const noexcept {
        return status_ == that.status_;
    }
};

class LabelParams : public BaseParams {
    GETTER(Lids, lids)

    YREFLECTION_GRANT_ACCESS
    LabelParams() = default;
public:
    explicit LabelParams(Lids lids)
            : BaseParams(),
              lids_(std::move(lids)) {
        if (lids_.empty()) {
            throw ParamsException("lids cannot be empty");
        }
    }

    bool operator== (const LabelParams& that) const noexcept {
        return lids_ == that.lids_;
    }
};

class UnlabelParams : public BaseParams {
    GETTER(Lids, lids)

    YREFLECTION_GRANT_ACCESS
    UnlabelParams() = default;
public:
    explicit UnlabelParams(Lids lids)
            : BaseParams(),
              lids_(std::move(lids)) {
        if (lids_.empty()) {
            throw ParamsException("lids cannot be empty");
        }
    }

    bool operator== (const UnlabelParams& that) const noexcept {
        return lids_ == that.lids_;
    }
};

class DeleteLabelParams : public BaseParams {
    GETTER(Lid, lid)

    YREFLECTION_GRANT_ACCESS
    DeleteLabelParams() = default;
public:
    explicit DeleteLabelParams(Lid lid)
        : BaseParams(),
          lid_(std::move(lid))
    {}

    bool breakOnEmptyMids() const override {
        return false;
    }

    bool operator== (const DeleteLabelParams& that) const noexcept {
        return lid_ == that.lid_;
    }
};

class DeleteFolderParams : public BaseParams {
    GETTER(Fid, fid)

    YREFLECTION_GRANT_ACCESS
    DeleteFolderParams() = default;
public:
    explicit DeleteFolderParams(Fid fid)
        : BaseParams(),
          fid_(std::move(fid))
    {}

    ResolveOptions resolveOptions(const MailboxMeta&, const MailboxOperParams&, YieldCtx) const override {
        const auto excludeSystemFolder = [](const macs::Folder& folder) {
            return folder.isSystem();
        };
        return ResolveOptions({}, boost::none, ExcludeFilters{ excludeSystemFolder });
    }

    bool breakOnEmptyMids() const override {
        return false;
    }

    bool operator== (const DeleteFolderParams& that) const noexcept {
        return fid_ == that.fid_;
    }
};

#undef GETTER

} // namespace

