#include <mailbox_oper/mailbox_meta.h>

#include <boost/range/algorithm_ext/erase.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/range/numeric.hpp>
#include <boost/algorithm/cxx11/any_of.hpp>

namespace mbox_oper {

template <typename Range>
Mids getMids(const Range& envelopes) {
    Mids res;
    boost::transform(envelopes, std::back_inserter(res), [&](const auto& envelope) {
        return envelope.mid();
    });
    return res;
}


MailboxMetaImpl::MailboxMetaImpl(macs::ServicePtr metadata, ContextPtr ctx)
    : metadata(std::move(metadata)),
      logger(getContextLogger(ctx)),
      context(std::move(ctx))
{}

Mids MailboxMetaImpl::getMids(const Mids& mids, YieldCtx yield, const OptStatus& excludeStatus) const {
    if (!excludeStatus) {
        return mids;
    }
    const Mids res = metadata->envelopes().query()
            .withoutStatus(mids)
            .status(*excludeStatus)
            .get(make_yield_context(yield));

    if (res.empty()) {
        LOGDOG_(logger, notice, log::message="mids not found", log::mids=mids,
                log::exclude_status=statusToString(excludeStatus));
    }
    return res;
}

Mids MailboxMetaImpl::getMids(const Tids& tids, const ResolveOptions& options, YieldCtx yield) const {
    const Mids resMids = resolveMids(Mids(), tids, options, yield);
    if (resMids.empty()) {
        LOGDOG_(logger, notice, log::message="tids not found", log::tids=tids,
                log::resolve_options=options);
    }
    return resMids;
}

Mids MailboxMetaImpl::getMids(const Mids& mids, const Tids& tids, const ResolveOptions& options, YieldCtx yield) const {
    const Mids resMids = resolveMids(mids, tids, options, yield);
    if (resMids.empty()) {
        LOGDOG_(logger, notice, log::message="mids or tids not found",
                log::mids=mids, log::tids=tids,
                log::resolve_options=options);
    }
    return resMids;
}

Mids MailboxMetaImpl::getMids(const Fid& fid, const FidFilter& filter, YieldCtx yield,
                              const OptStatus& excludeStatus) const {
    const bool fidExists = metadata->folders().getAllFoldersWithHidden(make_yield_context(yield)).exists(fid);

    if (!fidExists) {
        LOGDOG_(logger, notice, log::message="fid not found", log::fid=fid);
        return Mids();
    }
    auto query = metadata->envelopes().query().mids(fid);
    if (filter.age()) {
        query.age(filter.age().get());
    }
    if (filter.subject()) {
        query.subject(filter.subject().get());
    }
    if (filter.from()) {
        query.from(filter.from().get());
    }
    if (excludeStatus) {
        query.excludeStatus(*excludeStatus);
    }

    macs::error_code ec;
    auto res = query.get(make_yield_context(yield, ec));

    if (ec) {
        const auto msg = "error when getting mids by folder: " + ec.message();
        throw MailboxOperException(msg);
    }
    return res;
}

Mids MailboxMetaImpl::getMids(const Fid& fid, const FidFilter& filter, const std::optional<Mid>& fromMid,
        std::size_t limit, YieldCtx yield, const OptStatus& excludeStatus) const {
    const bool fidExists = metadata->folders().getAllFoldersWithHidden(make_yield_context(yield)).exists(fid);

    if (!fidExists) {
        LOGDOG_(logger, notice, log::message="fid not found", log::fid=fid);
        return Mids();
    }
    auto query = metadata->envelopes().query().mids(fid);
    if (filter.age()) {
        query.age(filter.age().get());
    }
    if (filter.subject()) {
        query.subject(filter.subject().get());
    }
    if (filter.from()) {
        query.from(filter.from().get());
    }
    if (excludeStatus) {
        query.excludeStatus(*excludeStatus);
    }
    query.limit(limit);
    if (fromMid) {
        query.fromMid(*fromMid);
    }

    macs::error_code ec;
    auto res = query.get(make_yield_context(yield, ec));
    if (ec) {
        const auto msg = "error when getting mids range by folder: " + ec.message();
        throw MailboxOperException(msg);
    }
    return res;
}

Mids MailboxMetaImpl::getMids(const Lid& lid, YieldCtx yield) const {
    try {
        const auto label = metadata->labels().getAllLabels(make_yield_context(yield)).at(lid);
        return metadata->envelopes().query()
            .mids(std::move(label))
            .get(make_yield_context(yield));
    } catch (const boost::system::system_error& e) {
        if (e.code().value() == macs::error::noSuchLabel) {
            LOGDOG_(logger, notice, log::exception=e);
            return Mids();
        } else {
            throw;
        }
    }
}

Mids MailboxMetaImpl::getMids(const macs::Tab::Type& tab, YieldCtx yield,
                              const OptStatus& excludeStatus) const {
    const bool tabExists = metadata->tabs().getAllTabs(make_yield_context(yield)).exists(tab);

    if (!tabExists) {
        LOGDOG_(logger, notice, log::message="tab not found", log::tab=tab.toString());
        return Mids();
    }

    auto query = metadata->envelopes().query().mids(tab);
    if (excludeStatus) {
        query.excludeStatus(*excludeStatus);
    }

    macs::error_code ec;
    auto res = query.get(make_yield_context(yield, ec));

    if (ec) {
        const auto msg = "error when getting mids by tab: " + ec.message();
        throw MailboxOperException(msg);
    }
    return res;
}

Mids MailboxMetaImpl::getMidsInFolderCascade(const Fid& fid, const ExcludeFilters& excludeFilters, YieldCtx yield) const {
    try {
        const auto folders = selectFoldersCascade(fid, excludeFilters, yield);

        Mids res;
        for (const macs::Folder& folder : folders) {
            const Mids mids = getMids(Fid(folder.fid()), FidFilter(), yield);
            res.insert(res.end(), mids.begin(), mids.end());
        }
        return res;
    } catch (const boost::system::system_error& e) {
        if (e.code().value() == macs::error::noSuchFolder) {
            LOGDOG_(logger, notice, log::message="fid not found", log::fid=fid);
            return Mids();
        } else {
            throw;
        }
    }
}

Mids MailboxMetaImpl::getMidsByTidAndWithSameHeaders(const Tid& tid, YieldCtx yield) const {
    return metadata->envelopes().query()
            .midsByTidAndWithSameHeaders(tid)
            .get(make_yield_context(yield));
}

Mids MailboxMetaImpl::getMids(const macs::HdrDateAndMessageIdVec& hdrPairs, YieldCtx yield) const {
    return metadata->envelopes().query()
            .mids(hdrPairs)
            .get(make_yield_context(yield));
}

size_t MailboxMetaImpl::getMidsCount(const Fid& fid, const OptStatus& excludeStatus, YieldCtx yield) const {
    const auto folders = metadata->folders().getAllFoldersWithHidden(make_yield_context(yield));

    const auto& folder = folders.at(fid);
    if (excludeStatus == macs::Envelope::Status_read) {
        return folder.newMessagesCount();
    } else if (excludeStatus == macs::Envelope::Status_unread) {
        return folder.messagesCount() - folder.newMessagesCount();
    } else {
        return folder.messagesCount();
    }
}

size_t MailboxMetaImpl::getMidsCount(const Lid& lid, YieldCtx yield) const {
    const auto labels = metadata->labels().getAllLabels(make_yield_context(yield));
    const auto& label = labels.at(lid);
    return label.realMessagesCount().value_or(std::numeric_limits<size_t>::max());
}

size_t MailboxMetaImpl::getMidsCount(const macs::Tab::Type& tabType, const OptStatus& excludeStatus, YieldCtx yield) const {
    const auto tabs = metadata->tabs().getAllTabs(make_yield_context(yield));
    const auto& tab = tabs.at(tabType);
    if (excludeStatus == macs::Envelope::Status_read) {
        return tab.newMessagesCount();
    } else if (excludeStatus == macs::Envelope::Status_unread) {
        return tab.messagesCount() - tab.newMessagesCount();
    } else {
        return tab.messagesCount();
    }
}

size_t MailboxMetaImpl::getMessageCountInThreads(const Tids& tids, YieldCtx yield) const {
    const macs::TidVector tidsVector(tids.begin(), tids.end());
    return metadata->threads().getMessageCountInThreads(tidsVector, make_yield_context(yield));
}

size_t MailboxMetaImpl::getMidsCountCascade(const Fid& fid, const ExcludeFilters& excludeFilters, YieldCtx yield) const {
    try {
        const auto folders = selectFoldersCascade(fid, excludeFilters, yield);
        const auto getMessageCount = [](const auto& folder) {
            return folder.messagesCount();
        };

        using namespace boost;
        return accumulate(folders | adaptors::transformed(getMessageCount), 0u);
    } catch (const boost::system::system_error& e) {
        if (e.code().value() == macs::error::noSuchFolder) {
            LOGDOG_(logger, notice, log::message="fid not found", log::fid=fid);
            return 0u;
        } else {
            throw;
        }
    }
}

Fid MailboxMetaImpl::getFid(const macs::Folder::Symbol& symbol, YieldCtx yield) const {
    const auto folders = metadata->folders().getAllFoldersWithHidden(make_yield_context(yield));
    return Fid(folders.fid(symbol));
}

macs::Label MailboxMetaImpl::getLabel(const macs::Label::Symbol& symbol, YieldCtx yield) const {
    const auto labels = metadata->labels().getAllLabels(make_yield_context(yield));
    return labels.at(symbol);
}

macs::LabelSet MailboxMetaImpl::getLabelSet(YieldCtx yield) const {
    return metadata->labels().getAllLabels(make_yield_context(yield));
}

OptEnvelope MailboxMetaImpl::getEnvelope(const Mid& mid, YieldCtx yield) const {
    auto envelopes = metadata->envelopes().getByIds({ mid }, make_yield_context(yield));
    return envelopes.empty() ? OptEnvelope() : OptEnvelope(std::move(envelopes.front()));
}

Mids MailboxMetaImpl::selectMids(macs::ThreadMailboxItems items, const SkipFolders& skipFolders,
                                 YieldCtx yield) const {
    if (!skipFolders.empty()) {
        std::set<std::string> fids;
        const auto folders = metadata->folders().getAllFoldersWithHidden(make_yield_context(yield));

        boost::transform(skipFolders, std::inserter(fids, fids.end()),
                         [&](const auto& symbol) {
            return folders.fid(symbol);
        });

        boost::remove_erase_if(items, [&](const macs::ThreadMailboxItem& item) {
            const std::string& fid = item.fid;
            return fids.find(fid) != fids.end();
        });
    }

    return macs::getMids(items);
}

std::vector<macs::Folder> MailboxMetaImpl::selectFoldersCascade(const Fid& rootFid, const ExcludeFilters& excludeFilters,
                                                                YieldCtx yield) const {
    const macs::FolderSet allFolders = metadata->folders().getAllFoldersWithHidden(make_yield_context(yield));
    const macs::Folder& parentFolder = allFolders.at(rootFid);

    auto folders = allFolders.getChildrenRecursive(parentFolder);
    if (!excludeFilters.empty() && !folders.empty()) {
        boost::remove_erase_if(folders, [&](const macs::Folder& folder) {
            return boost::algorithm::any_of(excludeFilters, [&](const auto& filter) {
                return filter(folder);
            });
        });
    }
    folders.push_back(parentFolder);

    return folders;
}

Mids MailboxMetaImpl::resolveMids(const Mids& mids, const Tids& tids, const ResolveOptions& options, YieldCtx yield) const {
    const auto& skipFolders = options.skipFolders();
    const auto& excludeStatus = options.excludeStatus();

    if (!skipFolders.empty()) {
        macs::ThreadMailboxItems items;

        if (excludeStatus) {
            const auto selectedMids = metadata->threads().fillIdsListWithoutStatus(mids, tids, *excludeStatus, make_yield_context(yield));
            items = metadata->threads().fillIdsMap(selectedMids, Tids(), make_yield_context(yield));
        } else {
            items = metadata->threads().fillIdsMap(mids, tids, make_yield_context(yield));
        }

        return this->selectMids(items, skipFolders, yield);
    } else {
        if (excludeStatus) {
            return metadata->threads().fillIdsListWithoutStatus(mids, tids, *excludeStatus, make_yield_context(yield));
        } else {
            return metadata->threads().fillIdsList(mids, tids, make_yield_context(yield));
        }
    }
}

macs::settings::ParametersPtr MailboxMetaImpl::getParameters(std::vector<std::string> names, YieldCtx yield) const {
    return metadata->settings().getParameters(std::move(names), make_yield_context(yield));
}

} // namespace mbox_oper
