#include <mailbox_oper/mailbox_oper.h>
#include <mailbox_oper/logger.h>
#include <mailbox_oper/group_by.h>
#include <pgg/error.h>
#include <user_journal/parameters/message.h>
#include "../include/mailbox_oper/params.h"
#include <spdlog/details/format.h>


namespace {

void logChangeTabEvent(const user_journal::Journal& journal, const macs::ThreadMailboxItems& messages,
        std::optional<macs::Tab::Type> dstTab) {
    using namespace user_journal::parameters;

    if (!dstTab.has_value()) {
        return;
    }

    for (const auto& msg : messages) {
        const std::string srcTab = msg.tab.has_value() ? msg.tab->toString() : "";
        const std::string dstTabStr = dstTab->toString();
        journal.write<ChangeTab>(id::mdb("pg"),
                                  id::state(msg.mid),
                                  id::affected(1ul),
                                  id::mid(msg.mid),
                                  id::stid(msg.stid),
                                  id::receivedDate(msg.receivedDate),
                                  id::srcTab(srcTab),
                                  id::destTab(dstTabStr)
        );
    }
}

} // namespace

namespace mbox_oper {

struct TabMapping {
    const ContextLogger& logger_;
    std::map<macs::Lid, std::string> lidsToTypes_;

    static std::optional<macs::Lid> typeToLid(const macs::LabelSet& labels, const std::string& t) {
        const auto h = [=](const auto& v) {
            const macs::Label& label = v.second;
            return label.name() == t && label.type() == macs::Label::Type::spamDefense;
        };

        if(const auto it = boost::find_if(labels, h); it != labels.end()) {
            return it->second.lid();
        } else {
            return std::nullopt;
        }
    }

    TabMapping(const macs::deprecated::TabsMap& tabsMap,
               const ContextLogger& logger,
               const macs::LabelSet& labels)
        : logger_(logger)
    {
        for (auto it = tabsMap.getMap().begin(); it != tabsMap.getMap().end(); it++) {
            if (auto lid = typeToLid(labels, it->second); lid != std::nullopt) {
                lidsToTypes_[*lid] = it->first;
            }
        }
    }

    std::vector<macs::Lid> tabLids() const {
        std::vector<macs::Lid> ret;
        boost::copy(lidsToTypes_
                    | boost::adaptors::map_keys
                    , std::back_inserter(ret));

        return ret;
    }

    macs::ThreadMailboxItem setTabByLids(macs::ThreadMailboxItem item) {
        if (!item.tab && item.lidsToCheck.size() == 1) {
            item.tab = tabByLid(item.lidsToCheck[0]);
        } else if (item.lidsToCheck.size() > 1) {
            LOGDOG_(logger_, notice, log::message=fmt::format(
                "there are too many tab lids '{lids}' on mid '{mid}'",
                fmt::arg("lids", boost::algorithm::join(item.lidsToCheck, ",")),
                fmt::arg("mid", item.mid)
            ));
        }

        return item;
    }

    std::optional<macs::Tab::Type> tabByLid(const macs::Lid& lid) const {
        if (const auto it = lidsToTypes_.find(lid); it != lidsToTypes_.end()) {
            return macs::Tab::Type::fromString(it->second);
        } else {
            return std::nullopt;
        }
    }
};

MailboxOper::MailboxOper(macs::ServicePtr metadata,
                         std::shared_ptr<MailboxModifier> mailboxModifier,
                         ContextPtr context,
                         const user_journal::Journal& journal,
                         const macs::deprecated::TabsMap& tabsMap)
    : metadata_(std::move(metadata))
    , meta_(metadata_, context)
    , logger_(getContextLogger(context))
    , context_(std::move(context))
    , journal_(journal)
    , tabsMap_(tabsMap)
    , mailboxModifier_(mailboxModifier)
    {
}

void MailboxOper::spamMoveAndMark(const Mids& mids, const bool shouldMove,
                                  const std::string& spamFid, YieldCtx yield) {
    if (shouldMove && !mids.empty()) {
        modifyMailbox().moveMessages(spamFid, emptyTab, mids, yield);
    }

    exec(mids, MarkParams(macs::Envelope::Status_read), yield);
}

macs::ThreadMailboxItems MailboxOper::getMailboxItems(const Mids& mids, YieldCtx yield) {
    return getMailboxItems(mids, {}, yield);
}

macs::ThreadMailboxItems MailboxOper::getMailboxItems(const Mids& mids, const macs::Lids& lidsToCheck, YieldCtx yield) {
    if (mids.empty()) {
        return macs::ThreadMailboxItems();
    }

    const auto items = metadata().threads().fillIdsMap(mids, Tids(), lidsToCheck, make_yield_context(yield));

    if (items.empty()) {
        LOGDOG_(logger_, notice, log::message="elements not found", log::mids=mids);
    }
    return items;
}

void MailboxOper::exec(const Mids& mids, const MarkParams& params, YieldCtx yield) {
    const auto status = params.status();
    modifyMailbox().updateStatus(mids, status, yield);
    if (status == macs::Envelope::Status::Status_read) {
        macs::LabelSet allLabels = metadata().labels().getAllLabels(make_yield_context(yield));
        auto mul = allLabels.find(macs::Label::Symbol::mention_unvisited_label);
        if (mul != allLabels.end()) {
            modifyMailbox().changeEnvelopesLabels(mids, {}, {mul->second}, yield);
        }
    }
}

void MailboxOper::exec(const Mids& mids, const PurgeParams&, YieldCtx yield) {
    modifyMailbox().remove(mids, yield);
}

macs::ThreadMailboxItems MailboxOper::getUnreadMailboxItems(const macs::ThreadMailboxItems& messages) const {
    const auto filter = boost::adaptors::filtered([&](const auto& msg) {
        return !msg.seen;
    });

    macs::ThreadMailboxItems res;
    boost::copy(messages | filter, std::back_inserter(res));
    return res;
}

void MailboxOper::exec(const Mids& mids, const SpamParams& params, YieldCtx yield) {
    const macs::ThreadMailboxItems messages = getMailboxItems(mids, yield);
    exec(messages, params, yield);
}

void MailboxOper::exec(const macs::ThreadMailboxItems& messages, const SpamParams& params, YieldCtx yield) {
    if (messages.empty()) {
        return;
    }

    const auto spamFid = meta_.getFid(macs::Folder::Symbol::spam, yield);
    spamMoveAndMark(macs::getMids(messages), params.shouldMove(), spamFid, yield);
    logAbuse(messages, "spam", false);

    logChangeTabEvent(journal_, messages, std::nullopt);
}

void MailboxOper::exec(const Mids& mids, const UnspamParams& params, YieldCtx yield) {
    const macs::ThreadMailboxItems messages = getMailboxItems(mids, yield);
    if (!messages.empty()) {
        exec(messages, params, yield);
    }
}

void MailboxOper::exec(const macs::ThreadMailboxItems& messages, const UnspamParams& params,
        YieldCtx yield) {
    const auto destFid = params.destFid().empty() ?
                meta_.getFid(macs::Folder::Symbol::inbox, yield) :
                params.destFid();

    logAbuse(messages, "nonspam", false);

    if (params.shouldMove() && !messages.empty()) {
        const auto destTab = resolveDestTab(params.destTab());
        modifyMailbox().moveMessages(destFid, destTab, macs::getMids(messages), yield);
        logChangeTabEvent(journal_, messages, destTab);
    }
}

void MailboxOper::exec(const Mids& mids, const MoveParams& params, YieldCtx yield) {
    const macs::ThreadMailboxItems items = getMailboxItems(mids, yield);
    if (!items.empty()) {
        exec(items, params, yield);
    }
}

void MailboxOper::exec(const macs::ThreadMailboxItems& messages, const MoveParams& params, YieldCtx yield) {
    if (messages.empty()) {
        return;
    }

    auto& dstFid = params.destFid();
    auto& dstTab = params.destTab();

    try {
        const auto folder = metadata().folders()
                .getAllFoldersWithHidden(make_yield_context(yield)).at(dstFid);

        const auto destTab = resolveDestTab(dstTab);

        const auto mids = macs::getMids(messages);

        modifyMailbox().moveMessages(dstFid, destTab, mids, yield);

        if (folder.isTrash()) {
            using namespace user_journal::parameters;
            journal_.write<TrashMessages>(id::mdb("pg"),
                id::state(boost::algorithm::join(mids, ",")),
                id::affected(mids.size()), id::mids(mids.begin(), mids.end()));
        }

        logChangeTabEvent(journal_, messages, destTab);
    } catch (const boost::system::system_error& e) {
        if (e.code() == macs::error::noSuchFolder) {
            throw ParamsException("move to non-existent folder " + std::string(dstFid));
        } else if (e.code() == macs::error::noSuchTab) {
            throw ParamsException("move to non-existent tab " + dstTab.value_or(""));
        } else if (e.code() == pgg::errc::databaseLockFailed) {
            throw DatbaseLockFailed(e.code().message());
        } else {
            throw MailboxOperException("move error: " + e.code().message());
        }
    }
}

void MailboxOper::exec(const Mids& mids, const ComplexMoveParams& params, YieldCtx yield) {
    const auto& destFid = params.destFid();
    const auto mailboxSet = splitComplexMoveMessages(mids, destFid, yield);
    if (!mailboxSet.spamMailboxItems.empty()) {
        const auto spamParams = SpamParams(true, params.withSent());
        exec(mailboxSet.spamMailboxItems, spamParams, yield);
    }
    if (!mailboxSet.trashMailboxItems.empty()) {
        exec(mailboxSet.trashMailboxItems, TrashParams(), yield);
    }

    const auto tabKey = [] (const macs::ThreadMailboxItem& item) {
        return item.tab;
    };
    if (!mailboxSet.moveMailboxItems.empty()) {
        if (params.destTab()) {
            exec(mailboxSet.moveMailboxItems, MoveParams(destFid, params.destTab()), yield);
        } else {
            for(const auto& [key, value]: groupBy(mailboxSet.moveMailboxItems, tabKey)) {
                OptString tab;
                if (key) {
                    tab = key->toString();
                }
                exec(value, MoveParams(destFid, tab), yield);
            }
        }
    }
    if (!mailboxSet.unspamMailboxItems.empty()) {
        if (params.destTab()) {
            const auto unspamParams = UnspamParams(destFid, true, params.destTab());
            exec(mailboxSet.unspamMailboxItems, unspamParams, yield);
        } else {
            for(const auto& [key, value]: groupBy(mailboxSet.unspamMailboxItems, tabKey)) {
                OptString tab;
                if (key) {
                    tab = key->toString();
                }
                const auto unspamParams = UnspamParams(destFid, true, tab);
                exec(value, unspamParams, yield);
            }
        }
    }
}

void MailboxOper::exec(const Mids& mids, const TrashParams&, YieldCtx yield) {
    const macs::ThreadMailboxItems items = getMailboxItems(mids, yield);
    if (!items.empty()) {
        trash(items, yield);
    }
}

void MailboxOper::exec(const macs::ThreadMailboxItems& items, const TrashParams&, YieldCtx yield) {
    if (!items.empty()) {
        trash(items, yield);
    }
}

void MailboxOper::exec(const Mids& mids, const RemoveParams& params, YieldCtx yield) {
    const auto& fid = params.fid();
    const MidsPair midsPair = fid ? splitRemoveMessages(mids, *fid, yield)
                                  : splitRemoveMessages(mids, yield);
    const Mids& purgeMidsList = midsPair.first;
    const Mids& trashMidsList = midsPair.second;

    if (!purgeMidsList.empty() && !params.trashOnly()) {
        exec(purgeMidsList, PurgeParams(), yield);
    }

    if (!trashMidsList.empty()) {
        exec(trashMidsList, TrashParams(), yield);
    }
}

void MailboxOper::trash(const macs::ThreadMailboxItems& messages, YieldCtx yield) {
    const macs::ThreadMailboxItems unreadMessages = getUnreadMailboxItems(messages);
    std::string trashFid = meta_.getFid(macs::Folder::Symbol::trash, yield);

    logAbuse(unreadMessages, "unread_trash", true);

    exec(messages, MoveParams(Fid(std::move(trashFid))), yield);
}

MailboxOper::ThreadMailboxItemsPair MailboxOper::splitMessagesByFid(macs::ThreadMailboxItems messages,
                                                                    const macs::Fid& fid) {
    return splitMessagesByFids(messages, {fid});
}

MailboxOper::ThreadMailboxItemsPair MailboxOper::splitMessagesByFids(macs::ThreadMailboxItems messages,
                                                                     const std::vector<macs::Fid>& fids) {
    const auto bound = boost::partition(messages, [&](const auto& item) {
        return std::find(fids.begin(), fids.end(), item.fid) != fids.end();
    });
    macs::ThreadMailboxItems fidMessages(std::make_move_iterator(messages.begin()),
                                         std::make_move_iterator(bound));
    macs::ThreadMailboxItems nonFidMessages(std::make_move_iterator(bound),
                                            std::make_move_iterator(messages.end()));
    return std::make_pair(std::move(fidMessages), std::move(nonFidMessages));
}

void MailboxOper::exec(const Mids& mids, const LabelParams& params, YieldCtx yield) {
    const auto& lids = params.lids();

    if (!mids.empty()) {
        std::list<macs::Label> labels;
        const auto labelsSet = metadata().labels().getAllLabels(make_yield_context(yield));

        boost::transform(lids, std::back_inserter(labels), [&](const auto& lid) {
            return labelsSet.at(lid);
        });

        if (!labels.empty()) {
            modifyMailbox().changeEnvelopesLabels(mids, labels, {}, yield);
        }
    }
}

void MailboxOper::exec(const Mids& mids, const UnlabelParams& params, YieldCtx yield) {
    const auto& lids = params.lids();

    if (!mids.empty()) {
        std::list<macs::Label> labels;
        const auto labelsSet = metadata().labels().getAllLabels(make_yield_context(yield));

        boost::transform(lids, std::back_inserter(labels), [&](const auto& lid) {
            return labelsSet.at(lid);
        });

        if (!labels.empty()) {
            modifyMailbox().changeEnvelopesLabels(mids, {}, labels, yield);
        }
    }
}

MidsPair MailboxOper::splitRemoveMessages(const Mids& mids, YieldCtx yield) {
    auto messages = getMailboxItems(mids, yield);
    if (messages.empty()) {
        return MidsPair();
    }

    const auto trashFid = meta_.getFid(macs::Folder::Symbol::trash, yield);
    const auto hiddenTrashFid = meta_.getFid(macs::Folder::Symbol::hidden_trash, yield);
    const auto trashNonTrashMessages = splitMessagesByFids(std::move(messages), {trashFid, hiddenTrashFid});
    const macs::ThreadMailboxItems& trashMessages = trashNonTrashMessages.first;
    const macs::ThreadMailboxItems& nonTrashMessages = trashNonTrashMessages.second;
    return MidsPair(macs::getMids(trashMessages), macs::getMids(nonTrashMessages));
}

MidsPair MailboxOper::splitRemoveMessages(const Mids& mids, const Fid& fid, YieldCtx yield) {
    const auto trashFid = meta_.getFid(macs::Folder::Symbol::trash, yield);
    const auto hiddenTrashFid = meta_.getFid(macs::Folder::Symbol::hidden_trash, yield);
    const bool isTrashFid = (trashFid == fid) || (hiddenTrashFid == fid);
    return isTrashFid ? MidsPair(mids, Mids()) : MidsPair(Mids(), mids);
}

ComplexMoveMailboxSet MailboxOper::splitComplexMoveMessages(const Mids& mids,
        const std::string& destFid, YieldCtx yield) {

    TabMapping mapping(tabsMap().getMap(), logger_, meta_.getLabelSet(yield));

    auto messages = getMailboxItems(mids, mapping.tabLids(), yield);
    if (messages.empty()) {
        return ComplexMoveMailboxSet();
    }

    ComplexMoveMailboxSet res;
    const auto spamFid = meta_.getFid(macs::Folder::Symbol::spam, yield);
    const auto trashFid = meta_.getFid(macs::Folder::Symbol::trash, yield);

    if (destFid == static_cast<const std::string&>(spamFid)) {
        res.spamMailboxItems = messages;
        return res;
    } else if (destFid == static_cast<const std::string&>(trashFid)) {
        res.trashMailboxItems = messages;
        return res;
    } else {
        const auto h = [&mapping] (macs::ThreadMailboxItem item) {
            return mapping.setTabByLids(std::move(item));
        };

        const auto spamHamMessages = splitMessagesByFid(std::move(messages), spamFid);

        boost::copy(spamHamMessages.first
                    | boost::adaptors::transformed(h)
                    , std::back_inserter(res.unspamMailboxItems));

        boost::copy(spamHamMessages.second
                    | boost::adaptors::transformed(h)
                    , std::back_inserter(res.moveMailboxItems));

        return res;
    }
}

void MailboxOper::exec(const Mids& mids, const DeleteLabelParams& params, YieldCtx yield) {
    const auto& lid = params.lid();

    if (!mids.empty()) {
        exec(mids, UnlabelParams({ lid }), yield);
    }

    try {
        modifyMailbox().deleteLabel(lid, yield);
    } catch (const boost::system::system_error& e) {
        if (e.code().value() == macs::error::noSuchLabel) {
            LOGDOG_(logger_, notice, log::message="can't delete label", log::lid=lid, log::exception=e);
        } else {
            throw;
        }
    }
}

void MailboxOper::exec(const Mids& mids, const DeleteFolderParams& params, YieldCtx yield) {
    const auto& fid = params.fid();
    try {
        if (!mids.empty()) {
            exec(mids, TrashParams(), yield);
        }
        const auto folders = metadata().folders().getAllFolders(make_yield_context(yield));
        const auto& folder = folders.at(fid);
        modifyMailbox().eraseCascade(folder, yield);
    } catch (const boost::system::system_error& e) {
        if (e.code().value() == macs::error::noSuchFolder) {
            LOGDOG_(logger_, notice, log::message="can't delete folder", log::fid=fid, log::exception=e);
        } else {
            throw;
        }
    }
}

inline ::macs::error_code checkParentFid(const ::macs::Service& metadata,
                                         const ::macs::Fid& parentFid,
                                         YieldCtx yield) {
    if (parentFid == ::macs::Folder::noParent) {
        return {};
    }
    const auto allFolders = metadata.folders().getAllFolders(make_yield_context(yield));
    const bool badParent = allFolders.exists(parentFid)
            && allFolders.at(parentFid).symbolicName() == ::macs::Folder::Symbol::inbox;
    if (badParent) {
        return ::macs::error_code(::macs::error::folderCantBeParent);
    }
    return {};
}

macs::Folder MailboxOper::createFolder(const std::string& name, const macs::Fid parentFid,
        macs::Folder::Symbol symbol, YieldCtx yield) {

    const auto ec = checkParentFid(metadata(), parentFid, yield);
    if (ec) {
        throw ::macs::system_error(ec,
                "can't create folder with parent " + parentFid);
    }

    return modifyMailbox().createFolder(name, parentFid, symbol, yield);
}

macs::Folder MailboxOper::getOrCreateFolder(const std::string& name, const macs::Fid parentFid,
        macs::Folder::Symbol symbol, YieldCtx yield) {

    const auto ec = checkParentFid(metadata(), parentFid, yield);
    if (ec) {
        throw ::macs::system_error(ec,
                "can't create folder with parent " + parentFid);
    }

    return modifyMailbox().getOrCreateFolder(name, parentFid, symbol, yield);
}

macs::Folder MailboxOper::getOrCreateFolderBySymbolWithRandomizedName(const std::string& name, const macs::Fid& parent,
        macs::Folder::Symbol symbol, bool forceRandomize, YieldCtx yield) {
    const auto ec = checkParentFid(metadata(), parent, yield);
    if (ec) {
        throw ::macs::system_error(ec,
                "can't create folder with parent " + parent);
    }
    return modifyMailbox().getOrCreateFolderBySymbolWithRandomizedName(name, parent, symbol, forceRandomize, yield);
}

void MailboxOper::updateFolder(const macs::Fid& fid, const boost::optional<std::string>& name,
        const boost::optional<macs::Fid>& parentFid, YieldCtx yield) {
    if (name || parentFid) {
        const auto folders =  metadata().folders().getAllFolders(make_yield_context(yield));

        auto folder = folders.at(fid);
        macs::FolderFactory factory(std::move(folder));
        if (name) {
            factory.name(name.get());
        }
        if (parentFid) {
            const auto ec = checkParentFid(metadata(), parentFid.get(), yield);
            if (ec) {
                throw ::macs::system_error(ec,
                        "can't move folder " + fid + " to " +  parentFid.get());
            }
            factory.parentId(parentFid.get());
        }

        modifyMailbox().updateFolder(factory.product(), yield);
    }
}

void MailboxOper::setPop3(const std::vector<macs::Fid>& fids, YieldCtx yield) {
    modifyMailbox().setPop3(fids, yield);
}

void MailboxOper::setSortOptions(const macs::Fid& fid, const macs::Fid& prevFid, YieldCtx yield) {
    modifyMailbox().setSortOptions(fid, prevFid, yield);
}

void MailboxOper::setFolderSymbol(const macs::Fid& fid, const std::string& symbol, YieldCtx yield) {
    const auto folder = metadata().folders().getAllFolders(make_yield_context(yield)).at(fid);

    if (symbol.empty()) {
        modifyMailbox().resetFolderSymbol(folder, yield);
    } else {
        const auto folderSymbol = macs::Folder::Symbol::getByTitle(symbol);
        if (folderSymbol == macs::Folder::Symbol::none) {
            throw ParamsException("no such symbol");
        }
        modifyMailbox().setFolderSymbol(folder, folderSymbol, yield);
    }
}

macs::Label MailboxOper::createLabel(const boost::optional<std::string>& symbol, YieldCtx yield) {
    const macs::Label::Symbol& sym = macs::Label::Symbol::getByTitle(*symbol);
    if (sym == macs::Label::Symbol::none) {
        throw ParamsException("unknown symbol: " + *symbol);
    }

    return modifyMailbox().createLabel(sym, yield);
}

macs::Label MailboxOper::getOrCreateLabel(const boost::optional<std::string>& symbol, YieldCtx yield) {
    const macs::Label::Symbol& sym = macs::Label::Symbol::getByTitle(*symbol);
    if (sym == macs::Label::Symbol::none) {
        throw ParamsException("unknown symbol: " + *symbol);
    }

    return modifyMailbox().getOrCreateLabel(sym, yield);
}

macs::Label MailboxOper::createLabel(
        const std::string& name, const std::string& color, const std::string& typeName, YieldCtx yield) {
    const auto type =  macs::Label::Type::getByTitle(typeName);
    return modifyMailbox().createLabel(name, color, type, yield);
}

macs::Label MailboxOper::getOrCreateLabel(
        const std::string& name, const std::string& color, const std::string& typeName, YieldCtx yield) {
    const auto type =  macs::Label::Type::getByTitle(typeName);
    return modifyMailbox().getOrCreateLabel(name, color, type, yield);
}

void MailboxOper::updateLabel(const macs::Lid& lid,
        const boost::optional<std::string>& color,
        const boost::optional<std::string>& name, YieldCtx yield) {
    const auto label = metadata().labels().getAllLabels(make_yield_context(yield)).at(lid);

    auto factory = macs::LabelFactory(label);
    if (color) {
        factory.color(*color);
    }
    if (name) {
        factory.name(*name);
    }

    modifyMailbox().updateLabel(factory.product(), yield);
}

std::optional<macs::Tab::Type> MailboxOper::resolveDestTab(const OptString& destTab) const {
    if (destTab) {
        auto tab = macs::Tab::Type::fromString(*destTab, std::nothrow);
        if (tab != macs::Tab::Type::unknown) {
            return std::make_optional(tab);
        }
    }
    return emptyTab;
}

void MailboxOper::logAbuse(const macs::ThreadMailboxItems& messages, const std::string& type, bool hidden) const {
    using namespace user_journal::parameters;
    std::vector<macs::Mid> mids;
    mids.reserve(messages.size());

    boost::transform(messages, std::back_inserter(mids), [] (auto&& val) { return val.mid; });

    journal_.write<AbuseMessage>(
        id::mdb("pg"), id::affected(mids.size()), id::mids(mids),
        id::abuseType(type), id::hidden(hidden), id::state("")
    );
}

} // namespace mailbox_oper
