#include <mailbox_oper/params.h>
#include <mailbox_oper/mailbox_meta.h>
#include <boost/algorithm/cxx11/all_of.hpp>

namespace mbox_oper {

inline bool isNumber(const std::string& str) {
    std::uint64_t res;
    return boost::conversion::try_lexical_convert(str, res);
}

template <typename Strings>
bool isAllNumbers(const Strings& strings) {
    return boost::algorithm::all_of(strings, isNumber);
}

static Mids extractSyncMids(const Envelopes& envelopes, const MailboxMeta& mailboxMeta, YieldCtx yieldCtx) {
    if (envelopes.empty()) {
        return Mids();
    }

    macs::HdrDateAndMessageIdVec hdrPairs;
    boost::transform(envelopes, std::back_inserter(hdrPairs),
                    [](const macs::Envelope& envelope) {
        return std::make_pair(envelope.date(), envelope.rfcId());
    });

    Mids sourceMids;
    boost::transform(envelopes, std::back_inserter(sourceMids),
                     [](const auto& envelope) { return envelope.mid(); });

    auto mids = mailboxMeta.getMids(hdrPairs, yieldCtx);

    std::set<Mid> midsSet;
    const auto inserter = std::inserter(midsSet, midsSet.end());
    std::move(sourceMids.begin(), sourceMids.end(), inserter);
    std::move(mids.begin(), mids.end(), inserter);

    Mids result;
    std::move(midsSet.begin(), midsSet.end(), std::back_inserter(result));
    return result;
}

bool MidsSource::isNeedAsyncResolve(const ResolveOptions&) const {
    return true;
}

bool MidsSource::isNeedPaginate() const {
    return false;
}

OptValidationError DirectMidsSource::validate() const {
    if (isAllNumbers(mids)) {
        return boost::none;
    } else {
        return "illegal mids found: " + joinIds(mids);
    }
}

Mids DirectMidsSource::resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                               YieldCtx yieldCtx) const {
    if (isNeedSyncByHdrMessageId(options)) {
        auto envelope = mailboxMeta.getEnvelope(mids.front(), yieldCtx);
        const auto envelopes = envelope ? Envelopes{ envelope.get() } : Envelopes();
        return extractSyncMids(envelopes, mailboxMeta, yieldCtx);
    } else {
        return mailboxMeta.getMids(mids, yieldCtx, options.excludeStatus());
    }
}

size_t DirectMidsSource::getUpperMidsCount(const MailboxMeta&, const ResolveOptions&, YieldCtx) const {
    return mids.size();
}

MidsSourceData DirectMidsSource::getData() const {
    MidsSourceData data;
    data.mids = mids;
    return data;
}

bool DirectMidsSource::isNeedAsyncResolve(const ResolveOptions& options) const {
    return bool(options.excludeStatus());
}

bool DirectMidsSource::isNeedSyncByHdrMessageId(const ResolveOptions& options) const {
    return options.syncByHdrMessageId() && mids.size() == 1;
}

MidsSourcePtr DirectMidsSource::paginate(const std::optional<Mid>&, std::size_t) const {
    return std::make_shared<DirectMidsSource>(mids);
}

OptValidationError TidsSource::validate() const {
    if (isAllNumbers(tids)) {
        return boost::none;
    } else {
        return "illegal tids found:" + joinIds(tids);
    }
}

Mids TidsSource::resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const {
    if (isNeedSyncByHdrMessageId(options)) {
        return mailboxMeta.getMidsByTidAndWithSameHeaders(tids.front(), yieldCtx);
    } else {
        return mailboxMeta.getMids(tids, options, yieldCtx);
    }
}

size_t TidsSource::getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions&, YieldCtx yieldCtx) const {
    return mailboxMeta.getMessageCountInThreads(tids, yieldCtx);
}

MidsSourceData TidsSource::getData() const {
    MidsSourceData data;
    data.tids = tids;
    return data;
}

bool TidsSource::isNeedSyncByHdrMessageId(const ResolveOptions& options) const {
    return options.syncByHdrMessageId() && tids.size() == 1;
}

MidsSourcePtr TidsSource::paginate(const std::optional<Mid>&, std::size_t) const {
    return std::make_shared<TidsSource>(*this);
}

OptValidationError MidsWithTidsSource::validate() const {
    std::ostringstream ss;
    if (!isAllNumbers(mids)) {
        ss << "illegal mids found: " << joinIds(mids) << "; ";
    }

    if (!isAllNumbers(tids)) {
        ss << "illegal tids found: " << joinIds(tids);
    }

    const auto string = ss.str();
    if (string.empty()) {
        return boost::none;
    } else {
        return string;
    }
}

Mids MidsWithTidsSource::resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const {
    return mailboxMeta.getMids(mids, tids, options, yieldCtx);
}

size_t MidsWithTidsSource::getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions&,
                                             YieldCtx yieldCtx) const {
    return mids.size() + mailboxMeta.getMessageCountInThreads(tids, yieldCtx);
}

MidsSourceData MidsWithTidsSource::getData() const {
    MidsSourceData data;
    data.mids = mids;
    data.tids = tids;
    return data;
}

MidsSourcePtr MidsWithTidsSource::paginate(const std::optional<Mid>&, std::size_t) const {
    return std::make_shared<MidsWithTidsSource>(*this);
}

OptValidationError FidSource::validate() const {
    if (isNumber(fid)) {
        return boost::none;
    } else {
        return "illegal fid found: " + static_cast<const std::string&>(fid);
    }
}

Mids FidSource::resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const {
    if (options.cascadeOptions().enable()) {
        return mailboxMeta.getMidsInFolderCascade(fid, options.cascadeOptions().excludeFilters(), yieldCtx);
    } else {
        return mailboxMeta.getMids(fid, filter, yieldCtx, options.excludeStatus());
    }
}

size_t FidSource::getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options,
                                    YieldCtx yieldCtx) const {
    if (options.cascadeOptions().enable()) {
        return mailboxMeta.getMidsCountCascade(fid, options.cascadeOptions().excludeFilters(), yieldCtx);
    } else {
        return mailboxMeta.getMidsCount(fid, options.excludeStatus(), yieldCtx);
    }
}

MidsSourceData FidSource::getData() const {
    MidsSourceData data;
    data.fid = fid;
    data.age = filter.age().get_value_or(MidsSourceData::Days(0u));
    data.subject = filter.subject().get_value_or("");
    data.from = filter.from().get_value_or("");
    return data;
}

bool FidSource::isNeedPaginate() const {
    return true;
}

MidsSourcePtr FidSource::paginate(const std::optional<Mid>& fromMid, std::size_t limit) const {
    return std::make_shared<PagedFidSource>(*this, fromMid, limit);
}

OptValidationError PagedFidSource::validate() const {
    if (auto er = FidSource::validate()) {
        return er;
    }
    if (fromMid && !isNumber(*fromMid)) {
        return "illegal fromMid found: " + *fromMid;
    }
    return boost::none;
}

Mids PagedFidSource::resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const {
    if (options.cascadeOptions().enable()) {
        return mailboxMeta.getMidsInFolderCascade(getFid(), options.cascadeOptions().excludeFilters(), yieldCtx);
    } else {
        return mailboxMeta.getMids(getFid(), getFilter(), fromMid, limit, yieldCtx, options.excludeStatus());
    }
}

MidsSourceData PagedFidSource::getData() const {
    MidsSourceData data = FidSource::getData();
    data.fromMid = fromMid.value_or("");
    data.limit = std::to_string(limit);
    return data;
}

OptValidationError LidSource::validate() const {
    return boost::none;
}

Mids LidSource::resolve(const MailboxMeta& mailboxMeta, const ResolveOptions&, YieldCtx yieldCtx) const {
    return mailboxMeta.getMids(lid, yieldCtx);
}

size_t LidSource::getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions&, YieldCtx yieldCtx) const {
    return mailboxMeta.getMidsCount(lid, yieldCtx);
}

MidsSourceData LidSource::getData() const {
    MidsSourceData data;
    data.lid = lid;
    return data;
}

MidsSourcePtr LidSource::paginate(const std::optional<Mid>&, std::size_t) const {
    return std::make_shared<LidSource>(*this);
}

OptValidationError TabSource::validate() const {
    if (tabType() != macs::Tab::Type::unknown) {
        return boost::none;
    } else {
        return "unknown tab: " + tabName;
    }
}

size_t TabSource::getUpperMidsCount(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const {
    return mailboxMeta.getMidsCount(tabType(), options.excludeStatus(), yieldCtx);
}

MidsSourceData TabSource::getData() const {
    MidsSourceData data;
    data.tabName = tabName;
    return data;
}

Mids TabSource::resolve(const MailboxMeta& mailboxMeta, const ResolveOptions& options, YieldCtx yieldCtx) const {
    return mailboxMeta.getMids(tabType(), yieldCtx, options.excludeStatus());
}

MidsSourcePtr TabSource::paginate(const std::optional<Mid>&, std::size_t) const {
    return std::make_shared<TabSource>(*this);
}

MidsSourcePtr toMidsSource(const MidsSourceData& data) {
    if (!data.mids.empty()) {
        if (!data.tids.empty()) {
            return std::make_shared<MidsWithTidsSource>(data.mids, data.tids);
        }

        return std::make_shared<DirectMidsSource>(data.mids);
    }

    if (!data.tids.empty()) {
        return std::make_shared<TidsSource>(data.tids);
    }

    if (!data.fid.empty()) {
        if (data.limit.empty()) {
            FidFilter filter(data.age.count(), data.subject, data.from);
            return std::make_shared<FidSource>(Fid(data.fid), std::move(filter));
        } else {
            FidFilter filter(data.age.count(), data.subject, data.from);
            std::optional<std::string> fromMid;
            if (!data.fromMid.empty()) {
                fromMid = data.fromMid;
            }
            return std::make_shared<PagedFidSource>(FidSource(Fid(data.fid), std::move(filter)),
                    fromMid, std::stoul(data.limit));
        }
    }

    if (!data.lid.empty()) {
        return std::make_shared<LidSource>(Lid(data.lid));
    }

    if (!data.tabName.empty()) {
        return std::make_shared<TabSource>(data.tabName);
    }

    throw MailboxOperException("Invalid MidsSourceData format: all fields are empty");
}

ResolveOptions ComplexMoveParams::resolveOptions(const MailboxMeta& mailboxMeta, const MailboxOperParams&,
                                                 YieldCtx yieldCtx) const {
    const auto spamFid = mailboxMeta.getFid(macs::Folder::Symbol::spam, yieldCtx);
    const auto trashFid = mailboxMeta.getFid(macs::Folder::Symbol::trash, yieldCtx);
    const bool isWithSentByDefault = destFid_ == spamFid || destFid_ == trashFid;
    SkipFolders skipFolders = { macs::Folder::Symbol::outbox };
    if (!withSent_.get_value_or(isWithSentByDefault)) {
        skipFolders.push_back(macs::Folder::Symbol::sent);
    }
    return ResolveOptions(std::move(skipFolders));
}

ResolveOptions MarkParams::resolveOptions(const MailboxMeta&, const MailboxOperParams&, YieldCtx) const {
    const bool isPgMdb = true;
    return ResolveOptions({}, status_, CascadeOptions(), isPgMdb);
}

}
