#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <tvm_guard/module.h>

#include <mail/mops/reply_later/http_api/include/response.h>
#include <mail/mops/reply_later/http_api/include/parse.h>

#include <mail/mops/reply_later/core/include/log.h>
#include <mail/mops/reply_later/core/include/types.h>
#include <mail/mops/reply_later/service/include/config.h>
#include <mail/mops/reply_later/service/include/request_context.h>
#include <mail/mops/reply_later/service/include/handlers/stickers.h>
#include <mail/mops/include/internal/server/request_context.h>
#include <mail/mops/include/internal/server/parse_params.h>

#include <mail/http_getter/client/include/module.h>
#include <mail/callmeback/client/config_reflection.h>

#include <mail/webmail/http_api_helpers/include/handler_helpers.h>

#include <mail/ymod_maildb/include/module.h>

#include <yamail/data/deserialization/ptree_reader.h>


namespace mops::reply_later {

template <class F>
auto handler(ConfigPtr config, F f) {
    return [=] (ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
        http_api::Context ctx{s->request()};
        CommonParams common = parseCommonParams(ctx);

        auto log = http_getter::withLog(config->getter->httpLogger(common.uid, common.requestId));
        auto httpClient = config->getter->create(*s->request(), log);

        ymod_maildb::ServiceParams serviceParams = ymod_maildb::parseServiceParams("mops", *s);
        ymod_maildb::UserJournalParams userJournalParams = ymod_maildb::parseUserJournalParams(*s);
        macs::ServicePtr service = config->maildb->service(serviceParams, userJournalParams,
                std::make_shared<MacsPgLog>(getContextLogger(common.uid, common.requestId)), macs::pg::masterOnly);

        RequestContext req(std::move(httpClient), common);
        return f(std::move(common), std::move(req), std::move(service), std::move(ctx), std::move(yield));
    };
}

namespace ydd = yamail::data::deserialization;
namespace ytt = yplatform::time_traits;

struct HttpApi: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        using http_api::findDependency;

        http_api::BindInfo<Response> info {
            .guarded=findDependency<tvm_guard::Module>(cfg, "dependencies.guard"),
            .server=findDependency<ymod_webserver::server>(cfg, "dependencies.server"),
            .attributes=boost::coroutines::attributes(cfg.get<unsigned>("coroutine_stack_size")),
            .reactor=findDependency<yplatform::reactor>(cfg, "dependencies.reactor")
        };

        auto config = std::make_shared<Config> (Config {
            .getter = findDependency<http_getter::TypedClientModule>(cfg, "dependencies.http_getter"),
            .maildb = findDependency<ymod_maildb::Module>(cfg, "dependencies.maildb"),
            .callmebackConfig=ydd::fromPtree<callmeback::ClientConfig>(cfg.get_child("callmeback.client_configuration")),
            .callbackHost=cfg.get<std::string>("callback_host"),
            .allowedInterval=macs::StickerAllowedInterval{
                .min=ytt::duration_cast<ytt::seconds>(cfg.get<ytt::duration>("allowed_interval.min")).count(),
                .max=ytt::duration_cast<ytt::seconds>(cfg.get<ytt::duration>("allowed_interval.max")).count()
            }
        });

        const http_api::LogArgsAsTskvConfig logArgsConfig {
            .tskvFormat="mail-mops-post-args-log",
            .argsLogger=yplatform::log::find("post_args", false)
        };

        bindPOST<CreateReplyLaterResult>(info, logArgsConfig, "/reply_later/create",
                handler(config, [=] (CommonParams common, RequestContext req, macs::ServicePtr service, http_api::Context ctx, boost::asio::yield_context yield) {
            return createReplyLater(std::move(common), parseCreateReplyLaterParams(ctx), std::move(req), std::move(service), std::move(config), yield);
        }));

        bindPOST<RemoveReplyLaterResult>(info, logArgsConfig, "/reply_later/remove",
                handler(config, [=] (CommonParams common, RequestContext req, macs::ServicePtr service, http_api::Context ctx, boost::asio::yield_context yield) {
            return removeReplyLater(std::move(common), parseRemoveReplyLaterParams(ctx), std::move(req), std::move(service), std::move(config), yield);
        }));

        bindPOST<CallbackReplyLaterResult>(info, logArgsConfig, "/reply_later/callback",
                handler(config, [=] (CommonParams common, RequestContext req, macs::ServicePtr service, http_api::Context ctx, boost::asio::yield_context yield) {
            return callbackReplyLater(std::move(common), parseCallbackReplyLaterParams(ctx), std::move(req), std::move(service), std::move(config), yield);
        }));

        bindPOST<UpdateReplyLaterResult>(info, logArgsConfig, "/reply_later/update",
                handler(config, [=] (CommonParams common, RequestContext req, macs::ServicePtr service, http_api::Context ctx, boost::asio::yield_context yield) {
            return updateReplyLater(std::move(common), parseUpdateReplyLaterParams(ctx), std::move(req), std::move(service), std::move(config), yield);
        }));

        bindPOST<ResetReplyLaterResult>(info, logArgsConfig, "/reply_later/reset",
                handler(config, [=] (CommonParams common, RequestContext req, macs::ServicePtr service, http_api::Context ctx, boost::asio::yield_context yield) {
            return resetReplyLater(std::move(common), parseResetReplyLaterParams(ctx), std::move(req), std::move(service), std::move(config), yield);
        }));

        LOGDOG_(getModuleLogger(), notice, log::message="mops::reply_later::HttpApi loaded");
    }
};

}

DEFINE_SERVICE_OBJECT(mops::reply_later::HttpApi)
