#include <mail/mops/reply_later/service/include/handlers/stickers.h>
#include <mail/mops/reply_later/service/include/errors.h>
#include <yamail/data/deserialization/json_reader.h>
#include <mail/mops/include/internal/uuid.h>

BOOST_FUSION_DEFINE_STRUCT((mops::reply_later), Callback400Response,
    (std::string, status)
)

namespace mops::reply_later {

namespace detail {

template<class Logger, class Yield>
void removeIncorrectStickers(macs::ServicePtr service, Logger& log, Yield yy) {
    const macs::MidVec mids = service->stickers().removeIncorrectReplyLaterStickers(yy);

    if (!mids.empty()) {
        LOGDOG_(log, error, log::message="remove incorrect stickers", log::stickers_mids=boost::join(mids, ","),
                log::strange_stickers_count=mids.size());
    }
}

yamail::expected<macs::Fid> getReplyLaterFid(const macs::Service& service, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    const macs::FolderSet folders = service.folders().getAllFolders(yy);

    const auto replyLaterIt = folders.find(macs::Folder::Symbol::reply_later);
    if (folders.end() == replyLaterIt) {
        return make_unexpected(ServiceError::replyLaterFolderDoesNotExist);
    }
    return replyLaterIt->second.fid();
}

std::optional<macs::ReplyLaterSticker> getReplyLaterStickerOfEnvelope(const macs::Mid& mid, const macs::Service& service,
        boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    const macs::ReplyLaterStickers stickers = service.stickers().getReplyLaterList(yy);
    const auto stickerIt = std::find_if(stickers.begin(), stickers.end(), [&] (const macs::ReplyLaterSticker& s) {
        return s.mid == mid;
    });

    if (stickerIt == stickers.end()) {
        return std::nullopt;
    }

    return *stickerIt;
}

template <typename Yield>
void unmarkIfMarked(const macs::Envelope& envelope, const std::vector<macs::Label::Symbol>& symbols,
        const macs::Service& service, Yield yield) {
    std::list<macs::Label> labelsForUnmark;
    for (auto& symbol : symbols) {
        const macs::Label label = service.labels().getOrCreateLabel(symbol, yield);
        if (envelope.hasLabel(label.lid())) {
            labelsForUnmark.emplace_back(std::move(label));
        }
    }
    if (!labelsForUnmark.empty()) {
        service.envelopes().unmarkEnvelopes({envelope.mid()}, labelsForUnmark, yield);
    }
}

callmeback::Request makeRequest(const macs::Mid& mid, const macs::Fid& fid, const std::time_t& date,
        const std::optional<macs::Tab::Type>& tab, const CommonParams common) {
    callmeback::Request req;
    req.path = "/reply_later/callback";
    req.getArgs.add("mid", mid);
    req.getArgs.add("uid", common.uid);
    req.getArgs.add("fid", fid);
    if (tab) {
        req.getArgs.add("tab", tab->toString());
    }
    req.getArgs.add("date", std::to_string(date));
    req.headers.add("Content-Type", "application/x-www-form-urlencoded");
    req.headers.add("X-Request-Id", common.requestId);

    req.retryArgs = callmeback::RetryParamsExp(60, 2).data();

    return req;
}

macs::Fid chooseDstFid(const macs::Fid& originalFid, const macs::FolderSet& folders) {
    if (folders.exists(originalFid)) {
        return originalFid;
    }
    return folders.fid(macs::Folder::Symbol::inbox);
}

std::string generateReplyLaterEventId(const macs::Mid& mid) {
    return mid + '_' + helpers::getUuid<std::string>();
}

}

YREFLECTION_DEFINE_ENUM_INLINE(CallmebackOperation,
    setCallback,
    removeCallback
)

YREFLECTION_DEFINE_ENUM_INLINE(RemindersType,
    replyLater
)

std::string toString(RemindersType type) {
    return std::string(yamail::data::reflection::to_string(type));
}

void runTransactional(macs::ServicePtr service, std::function<void(macs::ServicePtr, boost::asio::yield_context)> func,
        boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    auto f = [func, yield] (macs::ServicePtr transService, std::function<void(macs::error_code)> hook) {
        boost::asio::spawn(yield, [transService, hook, func] (boost::asio::yield_context yield) {
            macs::error_code ec;
            try {
                func(transService, yield);
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const boost::system::system_error& ex) {
                ec = macs::error_code(ex.code(), ex.what());
            } catch (const std::exception& ex) {
                ec = make_error(ServiceError::unexpectedException, ex.what());
            } catch (...) {
                ec = make_error(ServiceError::unexpectedException, "unexpected exception type");
            }
            hook(std::move(ec));
        });
    };
    service->runTransactional(f, yy);
}

yamail::expected<EmptyResult> addCallmebackEvent(RequestContext req, ConfigPtr cfg, const macs::Mid& mid, const macs::Fid& fid,
        const std::optional<macs::Tab::Type>& tab, const std::time_t& date, const CommonParams& common, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    callmeback::Request request = detail::makeRequest(mid, fid, date, tab, common);
    const std::string reminderDate = callmeback::Client::formatDate(date);
    bool successfulSet = false;

    callmeback::Client(cfg->callmebackConfig, cfg->callbackHost, *req.httpClient)
            .set(common.uid, detail::generateReplyLaterEventId(mid), RemindersType::replyLater, reminderDate, request, false)
            ->call(CallmebackOperation::setCallback, http_getter::withDefaultHttpWrap([&](yhttp::response) {
                successfulSet = true;
            }), yy);

    if (!successfulSet) {
        return make_unexpected(ServiceError::callmebackError);
    }

    return {};
}

yamail::expected<EmptyResult> createReplyLater(CommonParams common, CreateReplyLaterParams params,
        RequestContext req,  macs::ServicePtr service, ConfigPtr cfg, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    const macs::Envelope envelope = service->envelopes().getById(params.mid, yy);
    const macs::Fid replyLaterFid = detail::getReplyLaterFid(*service, yield).value_or_throw();
    if (replyLaterFid == envelope.fid()) {
        return make_unexpected(ServiceError::envelopeAlreadyInReplyLaterFolder);
    }

    auto callmebackCallResult = addCallmebackEvent(std::move(req), cfg, params.mid, envelope.fid(),
            envelope.tab(), params.date, common, yield);
    if (!callmebackCallResult) {
        return callmebackCallResult;
    }

    auto f = [envelope = std::move(envelope), params = std::move(params), common = std::move(common),
            req = std::move(req), cfg = std::move(cfg), replyLaterFid = std::move(replyLaterFid)]
            (macs::ServicePtr service, boost::asio::yield_context yield) {
        using macs::Label;
        const auto yy = io_result::make_yield_context(yield);
        const std::vector<Label::Symbol> symbols {Label::Symbol::pinned_label, Label::Symbol::reply_later_finished};
        detail::unmarkIfMarked(envelope, symbols, *service, yy);

        const Label replyLaterStarted = service->labels().getOrCreateLabel(Label::Symbol::reply_later_started, yy);
        detail::removeIncorrectStickers(service, req.logger, yy);
        service->stickers().createReplyLater(params.mid, envelope.fid(), params.date, cfg->allowedInterval, envelope.tab(), yy);
        service->envelopes().moveMessages(replyLaterFid, envelope.tab(), {params.mid}, yy);
        service->envelopes().markEnvelopes({params.mid}, {replyLaterStarted}, yy);
    };
    runTransactional(service, f, yield);

    return {};
}

yamail::expected<RemoveReplyLaterResult> removeReplyLater(CommonParams, RemoveReplyLaterParams params,
        RequestContext, macs::ServicePtr service, ConfigPtr, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    const std::optional<macs::ReplyLaterSticker> stickerOpt = detail::getReplyLaterStickerOfEnvelope(params.mid, *service, yield);
    if (!stickerOpt) {
        return make_unexpected(ServiceError::replyLaterStickerDoesNotExist);
    }

    auto f = [params, stickerOpt] (macs::ServicePtr service, boost::asio::yield_context yield) {
        const macs::Fid replyLaterFid = detail::getReplyLaterFid(*service, yield).value_or_throw();
        const auto yy = io_result::make_yield_context(yield);
        const macs::Envelope envelope = service->envelopes().getById(params.mid, yy);
        if (replyLaterFid == envelope.fid()) {
            const macs::Fid dstFid = detail::chooseDstFid(stickerOpt->fid, service->folders().getAllFolders(yy));
            service->envelopes().moveMessages(dstFid, stickerOpt->tab, {params.mid}, yy);
        }

        using macs::Label;
        const std::vector<Label::Symbol> symbols {Label::Symbol::pinned_label, Label::Symbol::reply_later_started,
                Label::Symbol::reply_later_finished};
        detail::unmarkIfMarked(envelope, symbols, *service, yy);
        service->stickers().removeReplyLater(params.mid, yy);
    };
    runTransactional(service, f, yield);

    return {};
}

bool compareCallbackParams(const CallbackReplyLaterParams& params, const macs::ReplyLaterSticker& sticker) {
    return params.mid == sticker.mid &&
        params.fid == sticker.fid &&
        params.tab.value_or(macs::Tab::Type::unknown) == sticker.tab.value_or(macs::Tab::Type::unknown) &&
        params.date == sticker.date;
}

yamail::expected<CallbackReplyLaterResult> callbackReplyLater(CommonParams, CallbackReplyLaterParams params,
        RequestContext req,  macs::ServicePtr service, ConfigPtr, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    mail_errors::error_code ec;
    const macs::Envelope envelope = service->envelopes().getById(params.mid, yy[ec]);
    if (ec == macs::error::noSuchMessage) {
        return make_unexpected(ServiceError::callbackEnvelopeDoesNotExist);
    } else if (ec) {
        return yamail::make_unexpected(ec);
    }

    const std::optional<macs::ReplyLaterSticker> stickerOpt = detail::getReplyLaterStickerOfEnvelope(params.mid, *service, yield);
    if (!stickerOpt) {
        return make_unexpected(ServiceError::callbackReplyLaterStickerDoesNotExist);
    }

    if (!compareCallbackParams(params, *stickerOpt)) {
        LOGDOG_(req.logger, warning, log::message=fmt::format("Got callback miss on mid: '{}', sticker date: '{}'", params.mid, stickerOpt->date));
        return make_unexpected(ServiceError::callbackMiss);
    }

    const macs::Fid replyLaterFid = detail::getReplyLaterFid(*service, yield).value_or_throw();
    if (replyLaterFid != envelope.fid()) {
        return make_unexpected(ServiceError::callbackEnvelopeIsNotInReplyLaterFolder);
    }

    using macs::Label;
    const Label replyLaterStarted = service->labels().getOrCreateLabel(Label::Symbol::reply_later_started, yy);
    if (!envelope.hasLabel(replyLaterStarted.lid())) {
        return make_unexpected(ServiceError::callbackEnvelopeHasNotReplyLaterStartedLabel);
    }

    auto f = [params = std::move(params), replyLaterStarted = std::move(replyLaterStarted), sticker = std::move(*stickerOpt)]
            (macs::ServicePtr service, boost::asio::yield_context yield) {
        const auto yy = io_result::make_yield_context(yield);
        const Label replyLaterFinished = service->labels().getOrCreateLabel(Label::Symbol::reply_later_finished, yy);
        const Label pinned = service->labels().getOrCreateLabel(Label::Symbol::pinned_label, yy);

        service->envelopes().markEnvelopes({params.mid}, {replyLaterFinished, pinned}, yy);
        const macs::Fid dstFid = detail::chooseDstFid(sticker.fid, service->folders().getAllFolders(yy));
        service->envelopes().moveMessages(dstFid, sticker.tab, {params.mid}, yy);
        service->envelopes().unmarkEnvelopes({params.mid}, {std::move(replyLaterStarted)}, yy);
    };
    runTransactional(service, f, yield);

    return {};
}

yamail::expected<UpdateReplyLaterResult> updateReplyLater(CommonParams common, UpdateReplyLaterParams params,
        RequestContext req, macs::ServicePtr service, ConfigPtr cfg, boost::asio::yield_context yield) {
    auto stickerOpt = detail::getReplyLaterStickerOfEnvelope(params.mid, *service, yield);
    if (!stickerOpt) {
        return make_unexpected(ServiceError::replyLaterStickerDoesNotExist);
    }
    if (stickerOpt->date == params.date) {
        return {};
    }

    const auto yy = io_result::make_yield_context(yield);
    const macs::Envelope envelope = service->envelopes().getById(params.mid, yy);

    const macs::Fid replyLaterFid = detail::getReplyLaterFid(*service, yield).value_or_throw();
    if (replyLaterFid != envelope.fid()) {
        return make_unexpected(ServiceError::callbackEnvelopeIsNotInReplyLaterFolder);
    }

    stickerOpt->date = params.date;
    auto callmebackCallResult = addCallmebackEvent(std::move(req), cfg, params.mid, stickerOpt->fid,
            stickerOpt->tab, params.date, common, yield);
    if (!callmebackCallResult) {
        return callmebackCallResult;
    }

    auto f = [params = std::move(params), envelope = std::move(envelope), common = std::move(common),
            req = std::move(req), cfg = std::move(cfg), sticker = std::move(*stickerOpt)]
            (macs::ServicePtr service, boost::asio::yield_context yield) {
        using macs::Label;
        const auto yy = io_result::make_yield_context(yield);
        const Label replyLaterStarted = service->labels().getOrCreateLabel(Label::Symbol::reply_later_started, yy);
        if (!envelope.hasLabel(replyLaterStarted.lid())) {
            LOGDOG_(req.logger, error, log::message=
                    fmt::format("Update sticker on mid '{}' without replyLaterStarted label", params.mid));
            service->envelopes().markEnvelopes({params.mid}, {std::move(replyLaterStarted)}, yy);
        }

        const Label replyLaterFinished = service->labels().getOrCreateLabel(Label::Symbol::reply_later_finished, yy);
        if (envelope.hasLabel(replyLaterFinished.lid())) {
            LOGDOG_(req.logger, error, log::message=
                    fmt::format("Update sticker on mid '{}' with replyLaterFinished label", params.mid));
        }

        const Label pinned = service->labels().getOrCreateLabel(Label::Symbol::pinned_label, yy);
        if (envelope.hasLabel(pinned.lid())) {
            LOGDOG_(req.logger, error, log::message=fmt::format("Update sticker on mid '{}' with pinned label", params.mid));
        }

        detail::unmarkIfMarked(envelope, {replyLaterFinished.symbolicName(), pinned.symbolicName()}, *service, yy);

        service->stickers().removeReplyLater(params.mid, yy);
        detail::removeIncorrectStickers(service, req.logger, yy);
        service->stickers().createReplyLater(params.mid, sticker.fid, params.date, cfg->allowedInterval, sticker.tab, yy);
    };
    runTransactional(service, f, yield);

    return {};
}

yamail::expected<ResetReplyLaterResult> resetReplyLater(CommonParams common, ResetReplyLaterParams params,
        RequestContext req, macs::ServicePtr service, ConfigPtr cfg, boost::asio::yield_context yield) {
    auto stickerOpt = detail::getReplyLaterStickerOfEnvelope(params.mid, *service, yield);
    if (!stickerOpt) {
        return make_unexpected(ServiceError::replyLaterStickerDoesNotExist);
    }

    const auto yy = io_result::make_yield_context(yield);

    const macs::Envelope envelope = service->envelopes().getById(params.mid, yy);
    const macs::Fid replyLaterFid = detail::getReplyLaterFid(*service, yield).value_or_throw();
    if (replyLaterFid == envelope.fid()) {
        return make_unexpected(ServiceError::envelopeAlreadyInReplyLaterFolder);
    }

    auto callmebackCallResult = addCallmebackEvent(std::move(req), cfg, params.mid, envelope.fid(),
            envelope.tab(), params.date, common, yield);
    if (!callmebackCallResult) {
        return callmebackCallResult;
    }

    auto f = [params = std::move(params), envelope = std::move(envelope), common = std::move(common),
            req = std::move(req), cfg = std::move(cfg), replyLaterFid = std::move(replyLaterFid)]
            (macs::ServicePtr service, boost::asio::yield_context yield) {
        const auto yy = io_result::make_yield_context(yield);
        service->stickers().removeReplyLater(params.mid, yy);

        using macs::Label;
        const std::vector<Label::Symbol> symbols{Label::Symbol::pinned_label, Label::Symbol::reply_later_finished};
        detail::unmarkIfMarked(envelope, symbols, *service, yy);
        detail::removeIncorrectStickers(service, req.logger, yy);
        service->stickers().createReplyLater(params.mid, envelope.fid(), params.date, cfg->allowedInterval, envelope.tab(), yy);
        service->envelopes().moveMessages(replyLaterFid, envelope.tab(), {params.mid}, yy);

        const Label replyLaterStarted = service->labels().getOrCreateLabel(Label::Symbol::reply_later_started, yy);
        service->labels().getOrCreateLabel(Label::Symbol::reply_later_started, yy);
        service->envelopes().markEnvelopes({params.mid}, {replyLaterStarted}, yy);
    };
    runTransactional(service, f, yield);

    return {};
}

}
