#include <mailbox_oper/mailbox_oper.h>
#include <ymod_taskmaster/errors.hpp>
#include <internal/server/base_handler.h>
#include <internal/server/parse_params.h>
#include <macs/errors.h>

using namespace std::string_literals;

namespace mbox_oper {

void BaseHandler::process(ymod_webserver::response_ptr response) const {
    RequestContext request(std::move(response));

    const auto expectedMethod = request.request().method;
    if (expectedMethod != method()) {
        request.responseMethodNotAllowed(method());
        return;
    }

    spawn([request = std::move(request), this](boost::asio::yield_context yield) {
        this->process(std::move(request), yield);
    });
}

void BaseHandler::process(RequestContext request, YieldCtx yield) const {
    try {
        logArguments(request);
        execute(request, yield);
    } catch (const ParamsException& e) {
        const auto msg = "invalid arguments: "s + e.what();
        request.responseInvalidRequest(msg);
        logError(request, msg);
    } catch (const macs::ParamsException& e) {
        const auto msg = "invalid arguments: "s + e.what();
        request.responseInvalidRequest(msg);
        logError(request, msg);
    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch (const ymod_taskmaster::LimitException& e) {
        const auto msg = "per-user limit exceeded: "s + e.what();
        request.responseForbidden(msg);
        logError(request, msg);
    } catch (const boost::system::system_error& e) {
        const auto c = e.code();
        std::ostringstream s;
        s << "system_error category: " << c.category().name()
                << " code: " << c.value() << " message: " << e.what();
        const auto msg = s.str();
        using namespace macs::error;
        if( c == noSuchFolder || c == noSuchLabel || c == invalidArgument
                || c == folderAlreadyExists || c == cantModifyFolder
                || c == duplicateLabelSymbol || c == cantModifyLabel
                || c == duplicateLabelNameType ) {
            request.responseInvalidRequest(msg);
        } else {
            request.responseInternalError(msg);
        }
        logError(request, s.str());
    } catch (const macs::deprecated::UnknownTabException& e) {
        request.responseInvalidRequest(e.what());
    } catch (const std::exception& e) {
        const auto msg = "exception: "s + e.what();
        request.responseInternalError(msg);
        logError(request, msg);
    } catch (...) {
        const auto msg = "unknown error"s;
        request.responseInternalError(msg);
        logError(request, msg);
    }
}

void BaseHandler::logArguments(const RequestContext& req) const {
    const auto transform = boost::adaptors::transformed([](const auto& arg) {
        return arg.first + '=' + arg.second;
    });
    const auto& url = req.request().url;
    const auto args = boost::algorithm::join(url.params | transform, "&");
    LOGDOG_(getContextLogger(req.context()), notice, log::path=url.make_full_path(), log::args=args);
}

} // namespace
