
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/cxx11/any_of.hpp>
#include <internal/server/parse_params.h>
#include <internal/async_api.h>

namespace mbox_oper {

using namespace boost::algorithm;

template <typename Sequence>
bool hasEmptyElement(const Sequence& sequence) {
    return any_of(sequence, [](const auto& item) { return item.empty(); });
}

std::list<std::string> getList(const RequestContext& req, const std::string& key) {
    const std::string str = req.getArg(key);

    if (str.empty()) {
        return std::list<std::string>();
    }
    std::list<std::string> res;
    split(res, str, is_any_of(","));
    return res;
}

MidsSourcePtr createFidSource(Fid fid, const RequestContext& req) {
    const auto age = req.getArg<unsigned>("age", 0u);
    auto subject = req.getArg("subject");
    auto from = req.getArg("from");

    FidFilter filter(age, std::move(subject), std::move(from));
    return std::make_shared<FidSource>(std::move(fid), std::move(filter));
}

MidsSourcePtr parseMidsSource(const RequestContext& req) {
    auto mids = getList(req, "mids");
    auto tids = getList(req, "tids");
    auto fid = Fid(req.getArg("fid"));
    auto tab = req.getArg("tab");
    auto lid = Lid(req.getArg("lid"));

    if (hasEmptyElement(mids)) {
        throw ParamsException("mid cannot be empty");
    }

    if (hasEmptyElement(tids)) {
        throw ParamsException("tid cannot be empty");
    }

    int sourceCount = (!mids.empty() || !tids.empty())
            + (!fid.empty())
            + (!tab.empty())
            + (!lid.empty());

    if (sourceCount > 1) {
        throw ParamsException("Too many arguments: mids, tids, fid, tab and lid cannot be set together"
                              "Allowed parameters combinations: mids, tids, fid, mids+tids, tab, lid");
    }

    if (!mids.empty() && !tids.empty()) {
        return std::make_shared<MidsWithTidsSource>(std::move(mids), std::move(tids));
    }

    if (!mids.empty()) {
        return std::make_shared<DirectMidsSource>(std::move(mids));
    }

    if (!tids.empty()) {
        return std::make_shared<TidsSource>(std::move(tids));
    }

    if (!fid.empty()) {
        return createFidSource(std::move(fid), req);
    }

    if (!tab.empty()) {
        return std::make_shared<TabSource>(std::move(tab));
    }

    if (!lid.empty()) {
        return std::make_shared<LidSource>(std::move(lid));
    }

    throw ParamsException("mids or tids or fid or lid or tab not found");
}

std::string getStatus(const RequestContext& req) {
    return req.getArg("status");
}

Fid getDestFid(const RequestContext& req) {
    return Fid(req.getArg("dest_fid"));
}

OptBool getWithSent(const RequestContext& req) {
    return req.getArgOptionalBool("with_sent");
}

Lids getLids(const RequestContext& req) {
    auto l = getList(req, "lids");
    if (hasEmptyElement(l)) {
        throw ParamsException("lid cannot be empty");
    }
    return Lids(std::make_move_iterator(l.begin()), std::make_move_iterator(l.end()));
}

Fids getFids(const RequestContext& req) {
    auto fids = getList(req, "fids");
    if (hasEmptyElement(fids)) {
        throw ParamsException("fid cannot be empty");
    }
    return Fids(std::make_move_iterator(fids.begin()), std::make_move_iterator(fids.end()));
}

OptString getDestTab(const RequestContext& req) {
    return req.getArgOptional("dest_tab");
}

bool shouldMove(const RequestContext& req) {
    const auto arg = req.getArgOptionalBool("nomove");
    return arg ? !*arg : true;
}

MailboxOperParams parseCommonParams(const RequestContext& req) {
    MailboxOperParams res;
    res.uid = req.getArgOrThrow("uid");
    res.login = req.getArg("login");
    res.karma = req.getArg("karma");
    res.karmaStatus = req.getArg("karma_status");
    res.userAgent = req.getArg("ua");
    res.sessionInfo = req.getArg("session_info");
    res.yandexUidCookie = req.getArg("yandexuid");
    res.iCookie = req.getArg("icookie");
    res.source = req.getArg("source");
    res.connectionId = req.getArg("connection_id");
    if (res.connectionId.empty()) {
        res.connectionId = req.getArg("ora_connection_id");
    }

    res.requestId = req.requestId();
    res.remoteIp = req.realIp();
    res.testBuckets = req.getHeader("X-Yandex-ExpBoxes");
    res.enabledTestBuckets = req.getHeader("X-Yandex-EnabledExpBoxes");
    res.clientType = req.getHeader("X-Yandex-ClientType");
    res.clientVersion = req.getHeader("X-Yandex-ClientVersion");
    return res;
}

SpamParams ParamsTraits<SpamParams>::parse(const RequestContext& req) {
    return SpamParams(shouldMove(req), getWithSent(req));
}

UnspamParams ParamsTraits<UnspamParams>::parse(const RequestContext& req) {
    return UnspamParams(getDestFid(req), shouldMove(req), getDestTab(req));
}

PurgeParams ParamsTraits<PurgeParams>::parse(const RequestContext&) {
    return PurgeParams();
}

RemoveParams ParamsTraits<RemoveParams>::parse(const RequestContext& req) {
    const bool trashOnly = req.getArgOptionalBool("nopurge").get_value_or(false);
    auto fidArg = req.getArgOptional("fid");
    auto fid = fidArg ? boost::make_optional(Fid(std::move(*fidArg))) : boost::none;
    return RemoveParams(trashOnly, getWithSent(req), std::move(fid));
}

TrashParams ParamsTraits<TrashParams>::parse(const RequestContext&) {
    return TrashParams();
}

MoveParams ParamsTraits<MoveParams>::parse(const RequestContext& req) {
    return MoveParams(getDestFid(req), getDestTab(req));
}

ComplexMoveParams ParamsTraits<ComplexMoveParams>::parse(const RequestContext& req) {
    return ComplexMoveParams(getDestFid(req), getWithSent(req), getDestTab(req));
}

MarkParams ParamsTraits<MarkParams>::parse(const RequestContext& req) {
    const auto status = statusFromString(getStatus(req));
    return MarkParams(status);
}

LabelParams ParamsTraits<LabelParams>::parse(const RequestContext& req) {
    return LabelParams(getLids(req));
}

UnlabelParams ParamsTraits<UnlabelParams>::parse(const RequestContext& req) {
    return UnlabelParams(getLids(req));
}

DeleteLabelParams ParamsTraits<DeleteLabelParams>::parse(const RequestContext& req) {
    auto lid = Lid(req.getArgOrThrow("lid"));
    return DeleteLabelParams(std::move(lid));
}

DeleteFolderParams ParamsTraits<DeleteFolderParams>::parse(const RequestContext& req) {
    auto fid = Fid(req.getArgOrThrow("fid"));
    return DeleteFolderParams(std::move(fid));
}

} // namespace
