from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    AccessLogCount,
    AccessLogCountByFirstStatusDigit,
    AccessLogCountByPathAndFirstStatusDigit,
    AccessLogRequestTimeHist,
    AccessLogRequestTimeHistByPath,
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
    SupervisorLogRestartMeters,
)
import os
import yaml
import logging
import mail.mops.unistat.cpp.run as run
import mail.unistat.cpp.cython.logs as logs
from mail.unistat.cpp.cython.count_by_condition import (
    Counter,
    Condition,
    Signal,
)


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


SERVICES = [
    ['blackbox', 'blackbox'],
    ['conninfo', 'sharpei'],
    ['send', 'push'],
    ['callmeback', 'callmeback'],
]

MOPS_ENDPOINTS = [
    '/spam',
    '/unspam',
    '/purge',
    '/purge_hidden_trash',
    '/remove',
    '/complex_move',
    '/mark',
    '/label',
    '/unlabel',

    '/folders/create',
    '/folders/create_hidden_trash',
    '/folders/update',
    '/folders/update_symbol',
    '/folders/update_position',
    '/folders/delete',
    '/folders/update_pop3',

    '/reply_later/create',
    '/reply_later/remove',
    '/reply_later/callback',
    '/reply_later/update',
    '/reply_later/reset',

    '/labels/create',
    '/labels/update',
    '/labels/delete'
]


def make_access_log_meters(name_prefix, endpoints):
    meters = [
        AccessLogCount(name_prefix),
        AccessLogCountByFirstStatusDigit(name_prefix),
        AccessLogRequestTimeHist((0, 100, 300, 1000, 5000), 'access_log_request'),
    ]

    for endpoint in endpoints:
        meters += [
            AccessLogCountByPathAndFirstStatusDigit(endpoint, name_prefix),
            AccessLogRequestTimeHistByPath((0, 100, 300, 1000, 5000), endpoint, 'access_log_request'),
        ]

    return meters


def make_mops_http_client_log_meters(services):
    meters = [
        HttpClientHttpRequestTotalTimeHist((0, 100, 500, 1000, 5000), service[0], service[1] + "_time") for service in services
    ]
    meters += [
        HttpClientHttpRequestCountByStatus(service[0], "count_by_status_" + service[1]) for service in services
    ]

    return meters


def make_mops_log_meters():
    stickers_signals = [Signal(cond=Condition.AnyStr(), name='strange_stickers_count_summ')]
    return [
        Counter('strange_stickers_count', stickers_signals, [])
    ]


def make_supervisor_meters():
    return [
        SupervisorLogRestartMeters('supervisor')
    ]


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='path for supervisord.log')
    parser.add_argument('config')
    return parser.parse_args()


def make_mops_config(data):
    return MopsConfig(
        access_log=os.path.join(os.curdir, data['config']['log']['access']['sinks'][0]['path']),
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks'][0]['path']),
        mops_log=os.path.join(os.curdir, data['config']['log']['mops']['sinks'][0]['path']),
    )


MopsConfig = namedtuple('MopsConfig', (
    'access_log',
    'httpclient_log',
    'mops_log',
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.config) as f:
        config = make_mops_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    access_log = logs.AccessTskv([], make_access_log_meters("access_log", MOPS_ENDPOINTS), fast_forward, config.access_log)
    http_client_log = logs.HttpClientLog([], make_mops_http_client_log_meters(SERVICES), fast_forward, config.httpclient_log)
    mops_log = logs.TskvLog([], make_mops_log_meters(), fast_forward, config.mops_log)

    logs_list = [access_log, http_client_log, mops_log]
    if args.supervisorlog is not None:
        supervisor_log = logs.SupervisorLog(
            [],
            make_supervisor_meters(),
            fast_forward,
            args.supervisorlog
        )
        logs_list.append(supervisor_log)

    run.run(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()
