#include <yplatform/find.h>
#include <yplatform/log.h>
#include <mailbox_oper/module.h>

#include <ymod_mops_worker/thread_worker.h>

#include <mail/mops/include/internal/uuid.h>

#include <boost/coroutine/attributes.hpp>
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include <boost/uuid/uuid_io.hpp>

namespace mops_worker {

using namespace ymod_taskmaster;

ThreadWorker::ThreadWorker(const Configuration& config,
                           const ContextHolder& holder)
    : config_(config)
    , holder_(holder)
    , launchId_(mops::helpers::getUuid<std::string>())
{ }

bool ThreadWorker::waitToContinue(boost::asio::yield_context yield) const {
    return !config_.switchbox->waitExpectedValue(config_.pauseName, false, io_result::make_yield_context(yield));
}

void ThreadWorker::operator()(boost::asio::yield_context yield) {
    while (waitToContinue(yield) && !holder_.cancelled()) {
        const auto context = boost::make_shared<Context>();

        ChunkHandler handler = [this, context, yield](const ChunkData& data) {
            this->process(data, std::move(context), yield);
        };

        bool processedChunk = false;
        try {
            processedChunk = config_.taskMaster->getChunk(handler, context,
                                                          this->launchId_, yield);
        } catch( const std::exception& e ) {
            LOGDOG_(this->makeContextLogger(context), error,
                    log::message="worker got exception while processing another chunk",
                    log::exception=e
            );
        }
        if (!processedChunk) {
            boost::this_thread::sleep_for(config_.sleepTime);
        }
    }
}

void ThreadWorker::process(const ChunkData& data, ContextPtr context, YieldCtx yield) const {
    const auto& taskData = data.task.taskData;
    auto commonParams = taskData.commonParams;
    commonParams.connectionId += "-async";

    auto moperModule = yplatform::find<mbox_oper::MailboxOperModule>("mailbox_oper");
    auto mailboxOper = moperModule->api(commonParams, context, yield);

    const auto task = ymod_taskmaster::deserializeTask(taskData);
    const auto& mids = data.chunk.mids;
    const auto chunkId = data.chunk.id;

    const auto ctxLogger = makeContextLogger(context);

    LOGDOG_(ctxLogger, notice, log::message="process", log::chunk=std::cref(data));
    if (mids.empty() && task->breakOnEmptyChunk()) {
        LOGDOG_(ctxLogger, warning, log::message="empty chunk, will do nothing", log::chunk_id=chunkId);
        return;
    }

    try {
        task->execute(mailboxOper, mids, true, context, yield);
    } catch (const mbox_oper::RetriableException& e) {
        LOGDOG_(ctxLogger, error, log::message="retriable exception in chunk", log::chunk_id=chunkId, log::exception=e);
        throw;
    } catch (const mbox_oper::ReadOnlyMetabaseException& e) {
        LOGDOG_(ctxLogger, error, log::message="Metabase-RO exception in chunk", log::chunk_id=chunkId, log::exception=e);
        throw;
    } catch (const std::exception& e) {
        LOGDOG_(ctxLogger, error, log::message="exception in chunk", log::chunk_id=chunkId, log::exception=e);
    }
}

} // namespace
