#pragma once

#include <cstddef>

#include <boost/shared_ptr.hpp>

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-parameter"
#include <boost/range/any_range.hpp>
#pragma GCC diagnostic pop

#include <boost/range/algorithm/copy.hpp>
#include <boost/variant.hpp>

#include <internal/user.hpp>
#include <ymod_taskmaster/types.hpp>

namespace ymod_taskmaster {

using Chunk = ChunkData;
using Chunks = std::vector<Chunk>;
using OptChunk = boost::optional<Chunk>;

inline ChunkIds getChunkIds(const Chunks& chunks) {
    ChunkIds res(chunks.size());
    boost::copy(chunks |
        boost::adaptors::transformed(boost::bind(&ChunkData::chunk, _1)) |
        boost::adaptors::transformed(boost::bind(&ChunkInfo::id, _1)),
        res.begin());
    return res;
}

struct UserChunk {
    User user;
    Chunk chunk;

    UserChunk() = default;

    UserChunk(const User& user, const Chunk& chunk)
        : user(user), chunk(chunk)
    {}
};

inline bool isValid(const UserChunk& userChunk) {
    return isValid(userChunk.user) && isValid(userChunk.chunk);
}

class ChunkPathInfo {
public:
    ChunkPathInfo()
    {}

    ChunkPathInfo(const UserId& uid, const std::string& pathName)
        : uid_(uid)
    {
        parsePathName(pathName);
    }

    ChunkPathInfo(const UserId& uid, const ChunkId& chunkId, const std::string& filter)
        : uid_(uid), chunkId_(chunkId), filter_(filter)
    {}

    const UserId& uid() const {
        return uid_;
    }

    const ChunkId& chunkId() const {
        return chunkId_;
    }

    const std::string& filter() const {
        return filter_;
    }

    std::string chunkNodeName() const {
        return chunkId_ + '_' + filter_;
    }

    bool operator==(const ChunkPathInfo& other) const {
        return uid_ == other.uid_ && chunkId_ == other.chunkId_ && filter_ == other.filter_;
    }

private:
    UserId uid_;
    ChunkId chunkId_;
    std::string filter_;

    void parsePathName(const std::string& name) {
        const size_t endOfUuid = findEndOfUuid(name);
        chunkId_ = name.substr(0, endOfUuid);
        filter_ = name.substr( endOfUuid + 1 );
    }

    size_t findEndOfUuid(const std::string& name) {
        const std::size_t endOfUuid = name.find_last_of('_');
        if( endOfUuid == std::string::npos || endOfUuid + 1 == name.size() ) {
            throw std::runtime_error("Unexpected chunk node name: " + name);
        }
        return endOfUuid;
    }
};

using ChunkPathInfos = std::vector<ChunkPathInfo>;

struct TryNextChunk final {};
using ProcessChunkResult = boost::variant<Chunk, TryNextChunk>;

class ChunksRepository {
public:
    virtual ~ChunksRepository() = default;

    virtual ProcessChunkResult processChunk(const User& user, const ChunkId& chunkId, ChunkHandler handler,
                                            YieldCtx yieldCtx) = 0;

    virtual ChunkIds chooseChunkIds(const User& user, size_t limit, YieldCtx yieldCtx) = 0;
};

typedef boost::shared_ptr<ChunksRepository> ChunksRepositoryPtr;

}
