#pragma once

#include <yplatform/log.h>
#include <ymod_taskmaster/context.hpp>
#include <ymod_taskmaster/types.hpp>
#include <ymod_taskmaster/marshalling.hpp>
#include <pgg/profiling.h>
#include <pgg/logdog.h>
#include <logdog/logger.h>
#include <logdog/backend/yplatform_log.h>
#include <logdog/format/tskv.h>
#include <mail/mops/include/common/logdog.h>
#include <mail/pgg/include/pgg/logdog.h>
#include <internal/profiling.hpp>

namespace ymod_taskmaster {

namespace log {

using namespace ::logdog;
using namespace ::mops::logdog;

LOGDOG_DEFINE_ATTRIBUTE(ymod_taskmaster::Mids, mids)

LOGDOG_DEFINE_ATTRIBUTE(ymod_taskmaster::TaskId , task_id)
LOGDOG_DEFINE_ATTRIBUTE(ymod_taskmaster::TaskInfo, task_info)

LOGDOG_DEFINE_ATTRIBUTE(ymod_taskmaster::ChunkId, chunk_id)
LOGDOG_DEFINE_ATTRIBUTE(ymod_taskmaster::ChunkIds , chunk_ids)
LOGDOG_DEFINE_ATTRIBUTE(ymod_taskmaster::ChunksInfo, chunks_info)
LOGDOG_DEFINE_ATTRIBUTE(std::string, chunk_version)

LOGDOG_DEFINE_ATTRIBUTE(std::chrono::microseconds, ttl)
LOGDOG_DEFINE_ATTRIBUTE(std::string, launch_id)

constexpr static auto formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-ymod_taskmaster-log"));

} // namespace log

using PlatformLogger = yplatform::log::source;

inline PlatformLogger getPlatformLogger() {
    return PlatformLogger(YGLOBAL_LOG_SERVICE, "task_master");
}

inline auto getContextLogger(ContextPtr ctx) {
    return log::makeContextLogger(log::make_log(log::formatter, getPlatformLogger()), std::move(ctx));
}
inline auto getContextLogger(const std::string& name, const std::string& requestId) {
    return getContextLogger(boost::make_shared<Context>(name, requestId));
}

using ContextLogger = decltype(getContextLogger(nullptr));

inline auto getRequestLogger(const std::string& requestId) {
    return getContextLogger(boost::make_shared<Context>("", requestId));
}

inline auto getModuleLogger(const std::string& name) {
    return ::logdog::make_log(
        log::formatter,
        std::make_shared<yplatform::log::source>(
            yplatform::log::find(name, true)
        )
    );
}

inline auto getModuleLogger() {
    return getModuleLogger("task_master");
}

using ModuleLogger = decltype(getModuleLogger());

class PgPaProfiler : public profiling::Log {
public:
    PgPaProfiler(ContextPtr context);
    void write(const std::string& operation, const std::string& info,
               pgg::Duration duration) const override;
private:
    ContextPtr context;
};

using PgLogger = pgg::logging::TypedLog<ContextLogger>;

} // namespace ymod_taskmaster

namespace logdog::tskv {

template <>
struct to_tskv_impl<ymod_taskmaster::ChunksInfo> {
    template <typename Out>
    static void apply(Out& out, const char* key, const ymod_taskmaster::ChunksInfo &v) {
        out << ytskv::attr(key, ymod_taskmaster::toString(v));
    }
};

template <>
struct to_tskv_impl<ymod_taskmaster::TaskInfo> {
    template <typename Out>
    static void apply(Out& out, const char* key, const ymod_taskmaster::TaskInfo& v) {
        out << ytskv::attr(key, ymod_taskmaster::toString(v));
    }
};

template <>
struct to_tskv_impl<ymod_taskmaster::Mids> {
    template <typename Out>
    static void apply(Out& out, const char* key, const ymod_taskmaster::Mids& v) {
        out << ytskv::attr(key, "[" + ymod_taskmaster::toString(v) + "]");
    }
};

} // namespace logdog::tskv
