#pragma once

#include <pgg/profiling.h>
#include <boost/exception/diagnostic_information.hpp>

namespace ymod_taskmaster::profiling {

using pgg::profiling::Log;
using pgg::profiling::LogPtr;

namespace details {

inline void write(LogPtr log, const pgg::Time& start,
                  const std::string& operation, const std::string& info) {
    const pgg::Time stop(pgg::now());
    const pgg::Duration duration(stop - start);
    log->write( operation, info, duration);
}

template <typename Handler>
std::enable_if_t<std::is_void_v<std::result_of_t<Handler()>>, void>
        callWithProfiling(Handler&& h, LogPtr log, const std::string& operation,
                          const std::string& info, const pgg::Time& start) {
    h();
    write(std::move(log), start, operation, info);
}

template <typename Handler>
std::enable_if_t<!std::is_void_v<std::result_of_t<Handler()>>, std::result_of_t<Handler()>>
        callWithProfiling(Handler&& h, LogPtr log, const std::string& operation,
                          const std::string& info, const pgg::Time& start) {
    std::result_of_t<Handler()> retval = h();
    write(std::move(log), start, operation, info);
    return retval;
}


template <typename Handler>
decltype(auto) callWithProfiling(Handler&& h, LogPtr log,
                                 const std::string& operation, const std::string& info) {
    const pgg::Time start(pgg::now());
    try {
        return callWithProfiling(std::forward<Handler>(h), log, operation, info, start);
    } catch(const boost::exception& e) {
        write(log, start, operation, boost::diagnostic_information(e));
        throw;
    } catch(const std::exception& e) {
        write(log, start, operation, e.what());
        throw;
    } catch(...) {
        write(log, start, operation, "interrupted with unknown exception");
        throw;
    }
}

} // namespace details

template <typename Handler>
decltype(auto) call(Handler&& h, LogPtr log,
                    const std::string& operation, const std::string& info = "") {
    return log ? details::callWithProfiling(std::forward<Handler>(h), std::move(log), operation, info) : h();
}

} // namespace ymod_taskmaster::profiling
