#pragma once

#include <internal/user.hpp>
#include <internal/chunk.hpp>
#include <internal/tasks_storage_pg.hpp>
#include <ymod_taskmaster/errors.hpp>
#include <ymod_taskmaster/task.hpp>

namespace ymod_taskmaster {

class TasksRepository {
public:
    virtual ~TasksRepository() = default;

    virtual TaskGroupId push(const User& user, TaskPtr task, const Mids& mids, YieldCtx yieldCtx) = 0;

    virtual bool existsQueue(const User& user, YieldCtx yieldCtx) const = 0;

    virtual TaskStatParams readTask(const User& user, const TaskId& taskId, YieldCtx yieldCtx) = 0;
};

typedef boost::shared_ptr<TasksRepository> TasksRepositoryPtr;

class TasksRepositoryImpl: public TasksRepository {
public:
    TasksRepositoryImpl(RepositoryPtr pgRequest, ContextPtr context, const size_t chunkSize, const size_t tasksLimit)
        : pgRepo_(std::move(pgRequest), std::move(context)), chunkSize_(chunkSize), tasksLimit_(tasksLimit)
    {}

    TaskGroupId push(const User& user, TaskPtr task, const Mids& mids, YieldCtx yieldCtx) override {
        if (!pgRepo_.canPushTask(user, tasksLimit_, yieldCtx)) {
            throw LimitException("per-user tasks limit reached for uid=" + user.uid());
        }

        const auto taskChunks = task->createChunks(chunkSize_, mids);
        const auto& taskInfo = taskChunks.taskInfo;

        pgRepo_.pushTask(user, taskChunks.chunks, taskInfo, yieldCtx);
        return taskInfo.taskGroupId();
    }

    bool existsQueue(const User& user, YieldCtx yieldCtx) const override {
        return pgRepo_.hasTasks(user, yieldCtx);
    }


    TaskStatParams readTask(const User& user, const TaskId& taskId, YieldCtx yieldCtx) override {
        return pgRepo_.readTask(user, taskId, yieldCtx);
    }

private:
    TasksStoragePg pgRepo_;
    size_t chunkSize_;
    size_t tasksLimit_;
};

} // namespace
