#pragma once

#include <ymod_taskmaster/types.hpp>

#include <boost/noncopyable.hpp>

namespace ymod_taskmaster {

class User {
public:
    User(const UserId& uid = UserId()) : uid_( uid )
    {}

    const UserId& uid() const {
        return uid_;
    }

    bool operator==(const User& other) const {
        return uid_ == other.uid_;
    }
private:
    UserId uid_;
};

typedef std::vector<User> Users;

inline bool isValid(const User& user) {
    return user.uid().size();
}

class UsersRepository {
public:
    virtual ~UsersRepository() {}

    virtual Users chooseUsers(size_t limit, YieldCtx yieldCtx) = 0;
    virtual UserStat userStat(const UserId& uid, YieldCtx yieldCtx) = 0;

private:
    virtual bool tryWorkerLock(const User& user, const std::string& launchId, YieldCtx yieldCtx) = 0;
    virtual void unlockWorker(const User& user, const std::string& launchId, YieldCtx yieldCtx) = 0;

    friend class WorkerUserTryLock;
};

typedef boost::shared_ptr<UsersRepository> UsersRepositoryPtr;

extern const std::string workerLockPrefix;

class WorkerUserTryLock: boost::noncopyable {
public:
    WorkerUserTryLock(UsersRepositoryPtr rep, const User& user,
                      const std::string& launchId, YieldCtx yieldCtx)
        : rep_(std::move(rep)), user_(user), launchId_(launchId), yieldCtx(yieldCtx) {
        relock();
    }

    ~WorkerUserTryLock() {
        rep_->unlockWorker(user_, launchId_, yieldCtx);
    }

    bool relock() {
        owns_ = rep_->tryWorkerLock(user_, launchId_, yieldCtx);
        return owns();
    }

    bool owns() const {
        return owns_;
    }

private:
    UsersRepositoryPtr rep_;
    const User& user_;
    const std::string& launchId_;
    YieldCtx yieldCtx;
    bool owns_;
};

}
