#pragma once

#include <mail/mops/include/internal/uuid.h>

#include <ymod_taskmaster/types.hpp>
#include <ymod_taskmaster/context.hpp>
#include <mailbox_oper/mailbox_oper.h>
#include <macs/tabs_map.h>

namespace ymod_taskmaster {

struct TaskChunks {
    ChunksData chunks;
    TaskInfo taskInfo;
};

class Task {
    mbox_oper::MailboxOperParams commonParams_;
    TaskId id_;
    TaskGroupId taskGroupId_;
public:
    explicit Task(mbox_oper::MailboxOperParams commonParams, TaskId taskId,
                  const TaskGroupId taskGroupId)
        : commonParams_(std::move(commonParams)),
          id_(std::move(taskId)),
          taskGroupId_(taskGroupId)
    {}

    virtual ~Task() = default;

    virtual void execute(mbox_oper::MailboxOper& mailboxOper, const Mids& mids,
                         const bool isAsync, ContextPtr context, YieldCtx yieldCtx) = 0;

    virtual TaskType type() const noexcept = 0;

    virtual TaskData data() const = 0;

    virtual TaskChunks createChunks(const size_t chunkSize, const Mids& mids) const = 0;

    virtual bool breakOnEmptyChunk() const = 0;

    const mbox_oper::MailboxOperParams& commonParams() const noexcept {
        return commonParams_;
    }

    const TaskId& id() const noexcept {
        return id_;
    }

    TaskGroupId taskGroupId() const noexcept {
        return taskGroupId_;
    }

    std::string userId() const noexcept {
        return commonParams_.uid;
    }

    void setId(TaskId id) {
        id_ = id;
    }
};

class OperationTask : public Task {
public:
    using Task::Task;

    virtual const mbox_oper::BaseParams& params() const noexcept = 0;

    virtual void setParams(TaskData& data) const = 0;
};

using TaskPtr = std::shared_ptr<Task>;
using OperationTaskPtr = std::shared_ptr<OperationTask>;

namespace detail {

template <typename Params>
class OperationTaskTemplate;

template <typename Params, TaskType TypeEnumValue>
struct BaseParamsTraits {
    using Task = OperationTaskTemplate<Params>;
    static constexpr TaskType taskType = TypeEnumValue;
};

template <typename Params>
struct ParamsTraits;

template <>
struct ParamsTraits<mbox_oper::SpamParams> : BaseParamsTraits<mbox_oper::SpamParams, TaskType::Spam> {
    static void setParams(TaskData& data, const mbox_oper::SpamParams& params) {
        data.spamParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::UnspamParams> : BaseParamsTraits<mbox_oper::UnspamParams, TaskType::Unspam> {
    static void setParams(TaskData& data, const mbox_oper::UnspamParams& params) {
        data.unspamParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::PurgeParams> : BaseParamsTraits<mbox_oper::PurgeParams, TaskType::Purge> {
    static void setParams(TaskData& data, const mbox_oper::PurgeParams& params) {
        data.purgeParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::TrashParams> : BaseParamsTraits<mbox_oper::TrashParams, TaskType::Trash> {
    static void setParams(TaskData& data, const mbox_oper::TrashParams& params) {
        data.trashParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::MarkParams> : BaseParamsTraits<mbox_oper::MarkParams, TaskType::Mark> {
    static constexpr TaskType taskType = TaskType::Mark;
    static void setParams(TaskData& data, const mbox_oper::MarkParams& params) {
        data.markParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::MoveParams> : BaseParamsTraits<mbox_oper::MoveParams, TaskType::Move> {
    static void setParams(TaskData& data, const mbox_oper::MoveParams& params) {
        data.moveParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::LabelParams> : BaseParamsTraits<mbox_oper::LabelParams, TaskType::Label> {
    static void setParams(TaskData& data, const mbox_oper::LabelParams& params) {
        data.labelParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::UnlabelParams> : BaseParamsTraits<mbox_oper::UnlabelParams, TaskType::Unlabel> {
    static void setParams(TaskData& data, const mbox_oper::UnlabelParams& params) {
        data.unlabelParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::DeleteLabelParams> : BaseParamsTraits<mbox_oper::DeleteLabelParams,
                                                                     TaskType::DeleteLabel> {
    static void setParams(TaskData& data, const mbox_oper::DeleteLabelParams& params) {
        data.deleteLabelParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::DeleteFolderParams> : BaseParamsTraits<mbox_oper::DeleteFolderParams,
                                                                      TaskType::DeleteFolder> {
    static void setParams(TaskData& data, const mbox_oper::DeleteFolderParams& params) {
        data.deleteFolderParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::RemoveParams> : BaseParamsTraits<mbox_oper::RemoveParams, TaskType::Remove> {
    static void setParams(TaskData& data, const mbox_oper::RemoveParams& params) {
        data.removeParams = params;
    }
};

template <>
struct ParamsTraits<mbox_oper::ComplexMoveParams> : BaseParamsTraits<mbox_oper::ComplexMoveParams,
                                                                     TaskType::ComplexMove> {
    static void setParams(TaskData& data, const mbox_oper::ComplexMoveParams& params) {
        data.complexMoveParams = params;
    }
};

template <typename Params>
class OperationTaskTemplate : public OperationTask {
    using Self = OperationTaskTemplate<Params>;
    using Traits = ParamsTraits<Params>;
protected:
    Params params_;
public:
    using ParamsType = Params;
    static constexpr TaskType typeValue = Traits::taskType;

    explicit OperationTaskTemplate(mbox_oper::MailboxOperParams commonParams,
                                   Params params, TaskId taskId = mops::helpers::getUuid<TaskId>(),
                                   const TaskGroupId taskGroupId = mops::helpers::getUuid<TaskGroupId>())
        : OperationTask(std::move(commonParams), std::move(taskId), taskGroupId),
          params_(std::move(params))
    {}

    TaskType type() const noexcept override {
        return typeValue;
    }

    void execute(mbox_oper::MailboxOper& mailboxOper, const Mids& mids,
                 const bool, ContextPtr, YieldCtx yieldCtx) override {
        mbox_oper::Operation<Params> operation{ convertMids(mids), params_ };
        mailboxOper.execute(operation, yieldCtx);
    }

    TaskData data() const override {
        TaskData data;
        data.type = type();
        data.id = id();
        data.taskGroupId = boost::uuids::to_string(taskGroupId());
        data.commonParams = commonParams();
        setParams(data);
        data.resolveMids = false;
        return data;
    }

    bool breakOnEmptyChunk() const override {
        return params_.breakOnEmptyMids();
    }

    TaskChunks createChunks(const size_t chunkSize, const Mids& mids) const override {
        const auto calcChunksCount = [&] {
            const auto midsCount = static_cast<double>(mids.size());
            const double chunksCount = midsCount / static_cast<double>(chunkSize);
            return size_t(ceil(chunksCount));
        };

        ChunksData chunks;
        TaskInfo taskInfo;
        taskInfo.chunksCount = calcChunksCount();
        taskInfo.taskData = data();
        taskInfo.creationSecs = toSeconds(now()).count();

        for (auto chunkBegin = mids.cbegin(); chunkBegin != mids.cend();) {
            const auto dist = static_cast<std::size_t>(std::distance(chunkBegin, mids.cend()));
            const std::size_t size = std::min(chunkSize, dist);
            const auto chunkEnd = chunkBegin + size;
            ChunkInfo chunkInfo{ mops::helpers::getUuid<ChunkId>(), Mids(chunkBegin, chunkEnd) };
            chunks.push_back(ChunkData{ std::move(chunkInfo), taskInfo });
            chunkBegin = chunkEnd;
        }

        return TaskChunks{ std::move(chunks), std::move(taskInfo) };
    }

    const mbox_oper::BaseParams& params() const noexcept override {
        return params_;
    }

    void setParams(TaskData& data) const override {
        Traits::setParams(data, params_);
    }
};

} // detail namespace

using SpamTask = detail::OperationTaskTemplate<mbox_oper::SpamParams>;
using UnspamTask = detail::OperationTaskTemplate<mbox_oper::UnspamParams>;
using PurgeTask = detail::OperationTaskTemplate<mbox_oper::PurgeParams>;
using TrashTask = detail::OperationTaskTemplate<mbox_oper::TrashParams>;
using MarkTask = detail::OperationTaskTemplate<mbox_oper::MarkParams>;
using MoveTask = detail::OperationTaskTemplate<mbox_oper::MoveParams>;
using LabelTask = detail::OperationTaskTemplate<mbox_oper::LabelParams>;
using UnlabelTask = detail::OperationTaskTemplate<mbox_oper::UnlabelParams>;
using RemoveTask = detail::OperationTaskTemplate<mbox_oper::RemoveParams>;
using ComplexMoveTask = detail::OperationTaskTemplate<mbox_oper::ComplexMoveParams>;

class DeleteLabelTask : public detail::OperationTaskTemplate<mbox_oper::DeleteLabelParams> {
public:
    using OperationTaskTemplate::OperationTaskTemplate;

    void execute(mbox_oper::MailboxOper& mailboxOper, const Mids& mids,
                 const bool isAsync, ContextPtr context, YieldCtx yieldCtx) override;
};

class DeleteFolderTask : public detail::OperationTaskTemplate<mbox_oper::DeleteFolderParams> {
public:
    using OperationTaskTemplate::OperationTaskTemplate;

    void execute(mbox_oper::MailboxOper& mailboxOper, const Mids& mids,
                 const bool isAsync, ContextPtr context, YieldCtx yieldCtx) override;
};

class ResolveMidsTask : public Task {
    mbox_oper::MidsSourcePtr source;
    OperationTaskPtr task;
public:
    explicit ResolveMidsTask(mbox_oper::MidsSourcePtr source, TaskId taskId, OperationTaskPtr task)
        : Task(task->commonParams(), std::move(taskId), task->taskGroupId()),
          source(std::move(source)),
          task(std::move(task))
    {}

    static TaskPtr deserialize(const TaskData& taskData);

    template <typename OpParams>
    explicit ResolveMidsTask(mbox_oper::MailboxOperParams commonParams, mbox_oper::MidsSourcePtr source,
                             OpParams params, TaskId taskId = mops::helpers::getUuid<TaskId>())
        : ResolveMidsTask(std::move(source), std::move(taskId),
                          std::make_shared<typename detail::ParamsTraits<OpParams>::Task>(
                              std::move(commonParams), std::move(params), mops::helpers::getUuid<TaskId>()))
    {}

    void execute(mbox_oper::MailboxOper& mailboxOper, const Mids&,
                 const bool isAsync, ContextPtr context, YieldCtx yieldCtx) override;

    TaskData data() const override;

    TaskType type() const noexcept override {
        return TaskType::ResolveMids;
    }

    TaskChunks createChunks(const size_t, const Mids&) const override;

    bool breakOnEmptyChunk() const override {
        return false;
    }

private:
    void planTask(TaskPtr task, const Mids& mids,
                  ContextPtr context, YieldCtx yieldCtx) const;
};

using ResolveMidsTaskPtr = std::shared_ptr<ResolveMidsTask>;

TaskPtr deserializeTask(const TaskData& taskData);

}
