#pragma once

#include <string>
#include <vector>
#include <list>

#include <boost/lexical_cast.hpp>
#include <boost/chrono.hpp>
#include <boost/property_tree/ptree_fwd.hpp>
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_io.hpp>
#include <mailbox_oper/params.h>
#include <pgg/request_info.h>

namespace ymod_taskmaster {

using Version = size_t;
using OptVersion = boost::optional<Version>;
static constexpr Version dataVersion = 2;

using boost::property_tree::ptree;

using Clock = boost::chrono::system_clock;
using Hours = boost::chrono::hours;
using Seconds = boost::chrono::seconds;
using Milliseconds = boost::chrono::milliseconds;
using Microseconds = boost::chrono::microseconds;
using Seconds = boost::chrono::seconds;
using Days = boost::chrono::duration<int, boost::ratio<86400>>;

using Duration = Microseconds;
using TimePoint = boost::chrono::time_point<Clock, Duration>;

using YieldCtx = mbox_oper::YieldCtx;

inline TimePoint now() {
    return boost::chrono::time_point_cast<Duration>(Clock::now());
}

inline Seconds toSeconds(const TimePoint& tp) {
    return boost::chrono::duration_cast<Seconds>(tp.time_since_epoch());
}

enum class TaskType {
    Unknown = -1,
    Spam = 0,
    Unspam = 1,
    Purge = 2,
    Trash = 3,
    Mark = 4,
    Move = 5,
    Label = 6,
    Unlabel = 7,
    DeleteLabel = 8,
    DeleteFolder = 9,
    Remove = 10,
    ComplexMove = 11,
    ResolveMids = 12
};

inline auto toValue(const TaskType type) noexcept {
    return static_cast<std::underlying_type_t<TaskType>>(type);
}

inline std::string toString(const TaskType type) {
    switch (type) {
        case TaskType::Spam:
            return "spam";
        case TaskType::Unspam:
            return "unspam";
        case TaskType::Purge:
            return "purge";
        case TaskType::Trash:
            return "trash";
        case TaskType::Mark:
            return "mark";
        case TaskType::Move:
            return "unmark";
        case TaskType::Label:
            return "label";
        case TaskType::Unlabel:
            return "unlabel";
        case TaskType::DeleteLabel:
            return "delete_label";
        case TaskType::DeleteFolder:
            return "delete_folder";
        case TaskType::Remove:
            return "remove";
        case TaskType::ComplexMove:
            return "complex_move";
        case TaskType::ResolveMids:
            return "resolve_mids";
        case TaskType::Unknown:
        default:
            return "unknown";
    }
}

using TaskId = std::string;
using TaskGroupId = boost::uuids::uuid;

using OptSpamParams = boost::optional<mbox_oper::SpamParams>;
using OptUnspamParams = boost::optional<mbox_oper::UnspamParams>;
using OptPurgeParams = boost::optional<mbox_oper::PurgeParams>;
using OptTrashParams = boost::optional<mbox_oper::TrashParams>;
using OptMarkParams = boost::optional<mbox_oper::MarkParams>;
using OptMoveParams = boost::optional<mbox_oper::MoveParams>;
using OptLabelParams = boost::optional<mbox_oper::LabelParams>;
using OptUnlabelParams = boost::optional<mbox_oper::UnlabelParams>;
using OptDeleteLabelParams = boost::optional<mbox_oper::DeleteLabelParams>;
using OptDeleteFolderParams = boost::optional<mbox_oper::DeleteFolderParams>;
using OptRemoveParams = boost::optional<mbox_oper::RemoveParams>;
using OptComplexMoveParams = boost::optional<mbox_oper::ComplexMoveParams>;

struct TaskData {
    mbox_oper::MailboxOperParams commonParams;
    TaskType type = TaskType::Unknown;
    TaskId id;
    std::string taskGroupId;
    mbox_oper::MidsSourceData source;
    bool resolveMids = false;
    TaskId nestedTaskId;
    OptSpamParams spamParams;
    OptUnspamParams unspamParams;
    OptPurgeParams purgeParams;
    OptTrashParams trashParams;
    OptMarkParams markParams;
    OptMoveParams moveParams;
    OptLabelParams labelParams;
    OptUnlabelParams unlabelParams;
    OptDeleteLabelParams deleteLabelParams;
    OptDeleteFolderParams deleteFolderParams;
    OptRemoveParams removeParams;
    OptComplexMoveParams complexMoveParams;

    bool operator== (const TaskData& that) const noexcept {
        return commonParams == that.commonParams
            && type == that.type
            && id == that.id
            && taskGroupId == that.taskGroupId
            && source == that.source
            && resolveMids == that.resolveMids
            && nestedTaskId == that.nestedTaskId
            && spamParams == that.spamParams
            && unspamParams == that.unspamParams
            && purgeParams == that.purgeParams
            && trashParams == that.trashParams
            && markParams == that.markParams
            && moveParams == that.moveParams
            && labelParams == that.labelParams
            && unlabelParams == that.unlabelParams
            && deleteLabelParams == that.deleteLabelParams
            && deleteFolderParams == that.deleteFolderParams
            && removeParams == that.removeParams
            && complexMoveParams == that.complexMoveParams;
    }
};

struct TaskInfo {
    OptVersion version = dataVersion;
    TaskData taskData;
    size_t creationSecs;
    size_t chunksCount;

    const TaskId& id() const noexcept {
        return taskData.id;
    }

    TaskGroupId taskGroupId() const noexcept {
        return boost::lexical_cast<boost::uuids::uuid>(taskData.taskGroupId);
    }

    std::string versionString() const {
        return version ? std::to_string(*version) : std::string("ancient");
    }

    bool operator== (const TaskInfo& that) const noexcept {
        return version == that.version
            && taskData == that.taskData
            && creationSecs == that.creationSecs
            && chunksCount == that.chunksCount;
    }
};

using Mid = std::uint64_t;
using Mids = std::vector<Mid>;

using UserId = std::string;
using ChunkId = std::string;
using ChunkIds = std::vector<ChunkId>;

struct ChunkInfo {
    ChunkId id;
    Mids mids;

    bool operator==(const ChunkInfo& other) const {
        return id == other.id && mids == other.mids;
    }
};

using ChunksInfo = std::vector<ChunkInfo>;

struct ChunkData {
    ChunkInfo chunk;
    TaskInfo task;

    OptVersion version() const noexcept {
        return task.version;
    }

    std::string versionString() const {
        return task.versionString();
    }

    bool operator==(const ChunkData& other) const noexcept {
        return chunk == other.chunk && task == other.task;
    }

    bool operator!=(const ChunkData& other) const noexcept {
        return !(*this == other);
    }

    bool isExpired(const Seconds& ttl) const {
        const auto currentTime = toSeconds(now());
        const auto creationTime = Seconds(task.creationSecs);
        return currentTime >= creationTime && (currentTime - creationTime >= ttl);
    }
};

inline bool isValid(const ChunkData& chunk) {
    return chunk.chunk.id.size() && chunk.version() == dataVersion;
}

using ChunkHandler = std::function<void(const ChunkData&)>;

using ChunksData = std::vector<ChunkData>;
using ChunksPortions = std::vector<ChunksData>;

inline ChunksInfo getChunksInfo(const ChunksData& chunksData) {
    ChunksInfo res(chunksData.size());
    boost::transform(chunksData, res.begin(), [](const auto& data) {
        return data.chunk;
    });
    return res;
}

class TaskStatParams {
public:
    const TaskId& taskId() const {
        return taskId_;
    }

    void setTaskId(const TaskId& taskId) {
        taskId_ = taskId;
    }

    std::string taskGroupId() const {
        return boost::uuids::to_string(taskGroupId_);
    }

    void setTaskGroupId(const TaskGroupId id) {
        taskGroupId_ = id;
    }

    std::string type() const {
        return ymod_taskmaster::toString(type_);
    }

    void setType(const TaskType type) {
        type_ = type;
    }

    int creationTimestamp() const {
        const Seconds secs = toSeconds(creationTime_);
        return static_cast<int>(secs.count());
    }

    void setCreationTimestamp(const size_t secs) {
        creationTime_ = TimePoint(Seconds(secs));
    }

    size_t remainedChunksCount() const {
        return remainedChunksCount_;
    }

    void setRemainedChunksCount(size_t c) {
        remainedChunksCount_ = c;
    }

    size_t totalChunksCount() const {
        return totalChunksCount_;
    }

    void setTotalChunksCount(const size_t c) {
        totalChunksCount_ = c;
    }

    bool operator==(const TaskStatParams& other) const {
        return taskId_ == other.taskId_
            && type_ == other.type_
            && creationTime_ == other.creationTime_
            && remainedChunksCount_ == other.remainedChunksCount_
            && totalChunksCount_ == other.totalChunksCount_;
    }

    std::string toString() const {
        std::ostringstream ss;
        ss << "taskId=" << taskId_
           << "taskGroupId=" << taskGroupId_
           << "type=" << ymod_taskmaster::toString(type_)
           << "creationTime=" << creationTime_
           << "remainedChunksCount=" << remainedChunksCount_
           << "totalChunksCount=" << totalChunksCount_;
        return ss.str();
    }

private:
    TaskId taskId_;
    TaskGroupId taskGroupId_;
    TaskType type_ = TaskType::Unknown;
    TimePoint creationTime_;
    size_t remainedChunksCount_ = 0;
    size_t totalChunksCount_ = 0;
};

inline bool isValid(const TaskStatParams& task) {
    return !task.taskId().empty();
}

using TasksStatParams = std::vector<TaskStatParams>;

struct UserStat {
    TasksStatParams tasks;

    bool operator==(const UserStat& other) const {
        return tasks == other.tasks;
    }
};

inline mbox_oper::Mids convertMids(const Mids& mids) {
    mbox_oper::Mids res;
    const auto mid2mid = &boost::lexical_cast<std::string, Mid>;
    res.resize(mids.size());
    std::transform(mids.begin(), mids.end(), res.begin(), mid2mid);
    return res;
}

inline Mids convertMids(const mbox_oper::Mids& mids) {
    Mids res;
    const auto mid2mid = &boost::lexical_cast<Mid, std::string>;
    res.resize(mids.size());
    std::transform(mids.begin(), mids.end(), res.begin(), mid2mid);
    return res;
}

inline pgg::RequestInfo getRequestInfo(const mbox_oper::MailboxOperParams& params) {
    return pgg::RequestInfo{ params.requestId, params.connectionId,
                             params.clientType, params.remoteIp };
}

}
