
#include <ymod_taskmaster/marshalling.hpp>
#include <mail/mops/ymod_taskmaster/ymod_db/include/internal/errors.h>
#include <mail/io_result/include/io_result/coro.h>
#include <internal/chunks_storage_pg.hpp>

namespace ymod_taskmaster {

ChunksStoragePg::ChunksStoragePg(RepositoryPtr pgRequest, ContextPtr context,
                                 const size_t maxRetries, Milliseconds retryDelay,
                                 Seconds chunkTtl)
    : pgRequest_(std::move(pgRequest)), logger_(getContextLogger(context)), context_(std::move(context)),
      maxRetries_(maxRetries), retryDelay_(retryDelay), chunkTtl_(chunkTtl)
{}


Chunks ChunksStoragePg::readChunksByMids(const User& user, const Mids& mids, YieldCtx yieldCtx) {
    return fromSequence<Chunks>(
        pgRequest_->readChunksByMids(user, mids, context_, io_result::make_yield_context(yieldCtx))
    );
}

ChunkIds ChunksStoragePg::chooseChunkIds(const User& user, size_t limit, YieldCtx yieldCtx) {
    return fromSequence<ChunkIds>(
        pgRequest_->chooseChunkIds(user, limit, context_, io_result::make_yield_context(yieldCtx))
    );
}

Chunks ChunksStoragePg::readChunks(const User& user, const ChunkIds& chunkIds, YieldCtx yieldCtx) {
    return fromSequence<Chunks>(
        pgRequest_->readChunks(user, chunkIds, context_, io_result::make_yield_context(yieldCtx))
    );
}

ProcessChunkResult ChunksStoragePg::processChunk(const User& user, const ChunkId& chunkId, ChunkHandler handler,
                                                 YieldCtx yieldCtx) {
    try {
        const Chunks chunks = readChunks(user, {chunkId}, yieldCtx);
        if (chunks.empty()) {
            LOGDOG_(logger_, warning, log::message="no chunks returned, will do nothing",
                    log::uid=user.uid(), log::chunk_id=chunkId);
            return Chunk();
        }
        if (chunks.size() != 1) {
            LOGDOG_(logger_, warning, log::message="expected single chunk, actual size",
                    log::uid=user.uid(), log::chunk_id=chunkId, log::actual_size=chunks.size());
        }

        auto chunk = chunks.front();

        if (chunk.version() == dataVersion) {
            handler(chunk);
            removeChunk(user, chunkId, yieldCtx);
            return chunk;
        } else if (chunk.version() > dataVersion) {
            LOGDOG_(logger_, notice, log::message="skip chunk with the version newer than expected",
                    log::uid=user.uid(), log::chunk_id=chunkId, log::chunk_version=chunk.versionString());
            return TryNextChunk{};
        } else {
            if (chunk.isExpired(chunkTtl_)) {
                LOGDOG_(logger_, notice, log::message="expired chunk with incompatible version will be removed",
                        log::uid=user.uid(), log::chunk_id=chunkId, log::chunk_version=chunk.versionString());
                removeChunk(user, chunkId, yieldCtx);
            } else {
                LOGDOG_(logger_, notice, log::message="skip unexpired chunk with incompatible version",
                        log::uid=user.uid(), log::chunk_id=chunkId, log::chunk_version=chunk.versionString());
            }
            return TryNextChunk{};
        }
    } catch (const std::exception& e) {
        throw ChunkPgException(std::string("exception in ") + __FUNCTION__ + ": " + e.what());
    }
}

void ChunksStoragePg::removeChunk(const User& user, const ChunkId& chunkId, YieldCtx yieldCtx) {
    for (size_t retries = 1; retries <= maxRetries_; ++retries) {
        try {
            pgRequest_->removeChunk(user, chunkId, context_, io_result::make_yield_context(yieldCtx));
            break;
        } catch (const std::exception& e) {
            LOGDOG_(logger_, notice, log::message="can't remove chunk from pg",
                    log::uid=user.uid(), log::chunk_id=chunkId, log::try_number=retries, log::exception=e);
            boost::this_thread::sleep_for(retryDelay_);
        }
    }
}

} // namespace
