#include <ymod_taskmaster/marshalling.hpp>
#include <internal/queue_impl.hpp>
#include <internal/users_repository_impl.hpp>
#include <internal/chunks_repository_impl.hpp>

#include <boost/range/algorithm/random_shuffle.hpp>

namespace ymod_taskmaster {

QueueImpl::QueueImpl(const QueueSettings& settings, UsersRepositoryPtr usersRepo,
        ChunksRepositoryPtr chunksRepo, TasksRepositoryPtr tasksRepo, ContextPtr context, NotifierPtr notifier)
    : settings(settings), usersRepo(std::move(usersRepo)), chunksRepo(std::move(chunksRepo)),
      tasksRepo(std::move(tasksRepo)), context(std::move(context)), notifier(std::move(notifier))
{}

bool QueueImpl::isAsync(size_t midsCount) const {
    return settings.mids_for_async < midsCount;
}

TaskGroupId QueueImpl::pushTask(TaskPtr task, const Mids& mids, YieldCtx yieldCtx) {
    const User user(task->userId());
    return tasksRepo->push(user, std::move(task), mids, yieldCtx);
}

bool QueueImpl::processChunk(ChunkHandler handler, const std::string& launchId, YieldCtx yieldCtx) {
    const UserChunk userChunk = chooseChunkAndHandle(std::move(handler), launchId, yieldCtx);
    if (isValid(userChunk)) {
        notifier->notify(userChunk.user, userChunk.chunk, yieldCtx);
        return true;
    } else {
        return false;
    }
}

UserChunk QueueImpl::chooseChunkAndHandle(ChunkHandler handler, const std::string& launchId, YieldCtx yieldCtx) {
    Users users = usersRepo->chooseUsers(settings.choose_users_limit, yieldCtx);
    boost::random_shuffle(users);
    for (const User& user: users) {
        WorkerUserTryLock userLock(usersRepo, user, launchId, yieldCtx);
        if (userLock.owns()) {
            ChunkIds chunkIds = chunksRepo->chooseChunkIds(user, settings.choose_chunks_limit, yieldCtx);
            if (!chunkIds.empty()) {
                boost::random_shuffle(chunkIds);
            }

            for (const auto& chunkId : chunkIds) {
                auto result = chunksRepo->processChunk(user, chunkId, handler, yieldCtx);
                if (result.type() == typeid(TryNextChunk)) {
                    continue;
                }
                if (!userLock.relock()) {
                    return UserChunk();
                }
                return UserChunk(user, std::move(boost::get<Chunk>(result)));
            }
        }
    }
    return UserChunk();
}

UserStat QueueImpl::userStat(const UserId& uid, YieldCtx yieldCtx) {
    return usersRepo->userStat(uid, yieldCtx);
}

TaskStatParams QueueImpl::readTask(const UserId& uid, const TaskId& taskId, YieldCtx yieldCtx) {
    return tasksRepo->readTask(uid, taskId, yieldCtx);
}

}
