#include <internal/service_impl.hpp>
#include <internal/config_reflection.hpp>
#include <yplatform/module_registration.h>
#include <yamail/data/deserialization/ptree.h>
#include <yamail/data/serialization/ptree.h>
#include <fstream>

namespace ymod_taskmaster {

void ServiceImpl::init(const ptree& __xml) {
    loadSettings( __xml );
}

void ServiceImpl::reload(const ptree& __xml) {
    loadSettings( __xml );
}

void ServiceImpl::start() {
}

void ServiceImpl::stop() {
}

void ServiceImpl::fini () {
}

std::string readStoken(const std::string& path) {
    try {
        std::ifstream file;
        file.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        file.open(path);
        std::string stoken;
        std::getline(file, stoken);
        return stoken;
    } catch (const std::exception& e) {
        throw std::runtime_error("Can't read key from file \"" + path + "\": " + e.what());
    }
}

void ServiceImpl::loadSettings(const ptree& config) {
    settings.queue = yamail::data::deserialization::fromPtree<QueueSettings>(config.get_child("queue"));
    settings.notify = yamail::data::deserialization::fromPtree<NotifySettings>(config.get_child("notify"));
    httpGetter = yplatform::find<http_getter::TypedClientModule, std::shared_ptr>(config.get<std::string>("dependencies.http_getter"));
    pgRepo = yplatform::find<ymod_taskmaster::Repository, std::shared_ptr>(config.get<std::string>("dependencies.mopsdb"));

    settings.notify.stoken = readStoken(settings.notify.stoken_path);
    settings.notify.endpoint = http_getter::TypedEndpoint::fromPtree(config.get_child("notify.endpoint"));

}

ServiceImpl::Result ServiceImpl::planTaskIfAsync(TaskPtr task, const size_t midsCount,
                                                 ContextPtr context,
                                                 YieldCtx yieldCtx) {
    QueuePtr queue = makeQueue(std::move(context));
    if(queue->isAsync(midsCount)) {
        return queue->pushTask(std::move(task), Mids(), yieldCtx);
    }

    return Result();
}

ServiceImpl::Result ServiceImpl::planTaskIfAsync(TaskPtr task, const Mids& mids,
                                                 ContextPtr context,
                                                 YieldCtx yieldCtx) {
    QueuePtr queue = makeQueue(std::move(context));
    if(queue->isAsync(mids.size())) {
        return queue->pushTask(std::move(task), mids, yieldCtx);
    }

    return Result();
}

TaskGroupId ServiceImpl::planTask(TaskPtr task, const Mids& mids,
                                  ContextPtr context, YieldCtx yieldCtx) {
    QueuePtr queue = makeQueue(std::move(context));
    return queue->pushTask(std::move(task), mids, yieldCtx);
}

TaskStatParams ServiceImpl::readTask(const UserId& uid, const TaskId& taskId,
                                     ContextPtr context,
                                     YieldCtx yieldCtx) {
    QueuePtr queue = makeQueue(std::move(context));
    return queue->readTask(uid, taskId, yieldCtx);
}

bool ServiceImpl::getChunk(ChunkHandler handler, ContextPtr context,
                           const std::string& launchId, YieldCtx yieldCtx) {
    QueuePtr queue = makeQueue(std::move(context));
    return queue->processChunk(std::move(handler), launchId, yieldCtx);
}

QueuePtr ServiceImpl::makeQueue(ContextPtr context) {
    auto notify = settings.notify;
    notify.httpClient = httpGetter->create(context, nullptr);
    return getQueue(
        settings.queue, pgRepo, std::move(context), std::move(notify));
}

UserStat ServiceImpl::getUserStat(const std::string& uid, ContextPtr context,
                                  YieldCtx yieldCtx) {
    return makeQueue(std::move(context))->userStat(uid, yieldCtx);
}

const QueueSettings& ServiceImpl::getQueueSettings() const {
    return settings.queue;
}

}

DEFINE_SERVICE_OBJECT(ymod_taskmaster::ServiceImpl)
