#include <ymod_taskmaster/task.hpp>
#include <ymod_taskmaster/service.hpp>
#include <yplatform/find.h>

#include <yamail/data/serialization/yajl.h>
#include <yamail/data/deserialization/yajl.h>

using namespace yamail::data::deserialization;
using namespace yamail::data::serialization;
using namespace mbox_oper;

namespace ymod_taskmaster {

static bool isLastChunk(const UserId& uid, const TaskId& taskId,
                        ContextPtr context, YieldCtx yieldCtx) {
    auto taskMaster = yplatform::find<Service>("task_master");
    const auto taskParams = taskMaster->readTask(uid, taskId, std::move(context), yieldCtx);
    return taskParams.remainedChunksCount() <= 1;
}

inline bool isFinalOperation(const bool isAsync, const UserId& uid, const TaskId& taskId,
                             ContextPtr context, YieldCtx yieldCtx) {
    return !isAsync ||
           (isAsync && isLastChunk(uid ,taskId, std::move(context), yieldCtx));
}

void DeleteLabelTask::execute(MailboxOper& mailboxOper, const Mids& mids, const bool isAsync,
                              ContextPtr context, YieldCtx yieldCtx) {
    auto moperMids = convertMids(mids);
    if (isFinalOperation(isAsync, commonParams().uid, id(),
                         std::move(context), yieldCtx)) {
        const DeleteLabelOperation operation{ std::move(moperMids), params_ };
        mailboxOper.execute(operation, yieldCtx);
    } else {
        UnlabelParams params({ params_.lid() });
        const UnlabelOperation operation{ std::move(moperMids), std::move(params) };
        mailboxOper.execute(operation, yieldCtx);
    }
}

void DeleteFolderTask::execute(MailboxOper& mailboxOper, const Mids& mids, const bool isAsync,
                               ContextPtr context, YieldCtx yieldCtx) {
    auto moperMids = convertMids(mids);

    if (isFinalOperation(isAsync, commonParams().uid, id(),
                         std::move(context), yieldCtx)) {
        const DeleteFolderOperation operation{ std::move(moperMids), params_ };
        mailboxOper.execute(operation, yieldCtx);
    } else {
        const TrashOperation operation{ std::move(moperMids), TrashParams() };
        mailboxOper.execute(operation, yieldCtx);
    }
}

void ResolveMidsTask::execute(MailboxOper& mailboxOper, const Mids&, const bool,
                              ContextPtr context, YieldCtx yieldCtx) {
    const auto options = task->params().resolveOptions(mailboxOper.meta(), task->commonParams(), yieldCtx);

    const auto mids = source->resolve(mailboxOper.meta(), options, yieldCtx);
    if (!mids.empty()) {
        if (!source->isNeedPaginate()) {
            planTask(task, convertMids(mids), context, yieldCtx);
        } else {
            auto taskMaster = yplatform::find<Service>("task_master");

            if (mids.size() == 1) {
                planTask(task, convertMids(mids), context, yieldCtx);
            } else {
                using ymod_taskmaster::ResolveMidsTask;

                mbox_oper::Mids midsToPlan;
                midsToPlan.assign(std::next(mids.rbegin()), mids.rend());
                planTask(task, convertMids(midsToPlan), context, yieldCtx);

                source = source->paginate(*mids.rbegin(), taskMaster->getQueueSettings().max_mids_to_resolve);

                task->setId(mops::helpers::getUuid<TaskId>());
                auto resolveTask = std::make_shared<ResolveMidsTask>(std::move(source), mops::helpers::getUuid<TaskId>(), task);

                planTask(std::move(resolveTask), Mids(), std::move(context), yieldCtx);
            }
        }
    }
}

TaskData ResolveMidsTask::data() const {
    TaskData data;
    data.resolveMids = true;
    data.source = source->getData();
    data.type = task->type();
    data.id = id();
    data.taskGroupId = boost::uuids::to_string(taskGroupId());
    data.nestedTaskId = task->id();
    data.commonParams = commonParams();
    task->setParams(data);
    return data;
}

TaskChunks ResolveMidsTask::createChunks(const size_t, const Mids&) const {
    TaskInfo taskInfo;
    taskInfo.chunksCount = 1;
    taskInfo.taskData = data();
    taskInfo.creationSecs = toSeconds(now()).count();

    ChunkData chunk{ ChunkInfo{ mops::helpers::getUuid<ChunkId>(), Mids() }, taskInfo };
    return TaskChunks{ ChunksData{ std::move(chunk) }, std::move(taskInfo) };
}

void ResolveMidsTask::planTask(TaskPtr task, const Mids& mids,
                               ContextPtr context, YieldCtx yieldCtx) const {
    auto taskMaster = yplatform::find<Service>("task_master");
    taskMaster->planTask(std::move(task), mids, std::move(context), yieldCtx);
}

template <typename TaskT>
OperationTaskPtr deserializeOperationTask(const TaskData& taskData,
                                          const typename TaskT::ParamsType& params,
                                          const bool isNested) {
    const auto& taskId = isNested ? taskData.nestedTaskId : taskData.id;
    const TaskGroupId taskGroupId = boost::lexical_cast<boost::uuids::uuid>(taskData.taskGroupId);
    return std::make_shared<TaskT>(taskData.commonParams, params, taskId, taskGroupId);
}

static OperationTaskPtr deserializeOperationTask(const TaskData& taskData, const bool isNested) {
    const auto type = taskData.type;
    switch (type) {
        case TaskType::Spam:
            return deserializeOperationTask<SpamTask>(taskData, *taskData.spamParams, isNested);
        case TaskType::Unspam:
            return deserializeOperationTask<UnspamTask>(taskData, *taskData.unspamParams, isNested);
        case TaskType::Purge:
            return deserializeOperationTask<PurgeTask>(taskData, *taskData.purgeParams, isNested);
        case TaskType::Trash:
            return deserializeOperationTask<TrashTask>(taskData, *taskData.trashParams, isNested);
        case TaskType::Mark:
            return deserializeOperationTask<MarkTask>(taskData, *taskData.markParams, isNested);
        case TaskType::Move:
            return deserializeOperationTask<MoveTask>(taskData, *taskData.moveParams, isNested);
        case TaskType::Label:
            return deserializeOperationTask<LabelTask>(taskData, *taskData.labelParams, isNested);
        case TaskType::Unlabel:
            return deserializeOperationTask<UnlabelTask>(taskData, *taskData.unlabelParams, isNested);
        case TaskType::DeleteLabel:
            return deserializeOperationTask<DeleteLabelTask>(taskData, *taskData.deleteLabelParams, isNested);
        case TaskType::DeleteFolder:
            return deserializeOperationTask<DeleteFolderTask>(taskData, *taskData.deleteFolderParams, isNested);
        case TaskType::Remove:
            return deserializeOperationTask<RemoveTask>(taskData, *taskData.removeParams, isNested);
        case TaskType::ComplexMove:
            return deserializeOperationTask<ComplexMoveTask>(taskData, *taskData.complexMoveParams, isNested);
        default:
            throw std::runtime_error("Can't deserialize unknown operation task with type: " +
                                     std::to_string(toValue(type)));
    }
}

TaskPtr ResolveMidsTask::deserialize(const TaskData& taskData) {
    auto source = toMidsSource(taskData.source);
    auto task = deserializeOperationTask(taskData, true);
    return TaskPtr(new ResolveMidsTask(std::move(source), taskData.id, std::move(task)));
}

TaskPtr deserializeTask(const TaskData& taskData) {
    if (taskData.resolveMids) {
        return ResolveMidsTask::deserialize(taskData);
    } else {
        return deserializeOperationTask(taskData, false);
    }
}

}
