#pragma once

#include <mail/mops/ymod_taskmaster/include/ymod_taskmaster/context.hpp>
#include <mail/mops/ymod_taskmaster/include/internal/chunk.hpp>
#include <mail/mops/ymod_taskmaster/include/internal/user.hpp>

#include <io_result/io_result.h>
#include <io_result/hooks.h>

namespace ymod_taskmaster {

using RequestId = std::string;
using onExecute = io_result::Hook<>;
using onBoolean = io_result::Hook<bool>;
using onReadChunks = io_result::Hook<io_result::Sequence<Chunk>>;
using onChooseChunkIds = io_result::Hook<io_result::Sequence<ChunkId>>;
using onChooseUsers = io_result::Hook<io_result::Sequence<User>>;
using onReadTasks = io_result::Hook<io_result::Sequence<TaskStatParams>>;

template<typename T>
inline T fromSequence(auto range) {
    return {std::make_move_iterator(range.begin()), std::make_move_iterator(range.end())};
}

class Repository {
public:
    virtual ~Repository() = default;

    template<class Context = io_result::sync_context>
    auto addTask(const User& u, const ChunksData& c, const TaskInfo& t, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, onExecute> init(yy);
        asyncAddTask(u, c, t, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto readChunks(const User& u, const ChunkIds& chIds, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, onReadChunks> init(yy);
        asyncReadChunks(u, chIds, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto removeChunk(const User& u, const ChunkId& chId, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, onExecute> init(yy);
        asyncRemoveChunk(u, chId, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto canAddTask(const User& u, size_t tasksLimit, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, onBoolean> init(yy);
        asyncCanAddTask(u, tasksLimit, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto chooseChunkIds(const User& u, size_t limit, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, onChooseChunkIds> init(yy);
        asyncChooseChunkIds(u, limit, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto readChunksByMids(const User& u, const Mids& mids, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, onReadChunks> init(yy);
        asyncReadChunksByMids(u, mids, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto chooseUsers(size_t limit, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, onChooseUsers> init(yy);
        asyncChooseUsers(limit, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto hasTasks(const User& u, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) {
        io_result::detail::init_async_result<Context, onBoolean> init(yy);
        asyncHasTasks(u, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto userStat(const User& u, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) {
        io_result::detail::init_async_result<Context, onReadTasks> init(yy);
        asyncUserStat(u, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto readTask(const User& u, const TaskId& taskId, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) {
        io_result::detail::init_async_result<Context, onReadTasks> init(yy);
        asyncReadTask(u, taskId, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto acquireLock(const User& u, const std::chrono::microseconds& timeout, const std::string& launchId, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) {
        io_result::detail::init_async_result<Context, onBoolean> init(yy);
        asyncAcquireLock(u, timeout, launchId, ctx, init.handler);
        return init.result.get();
    }
    template<class Context = io_result::sync_context>
    auto releaseLock(const User& u, const std::string& launchId, 
            const ContextPtr& ctx, Context yy = io_result::use_sync) {
        io_result::detail::init_async_result<Context, onBoolean> init(yy);
        asyncReleaseLock(u, launchId, ctx, init.handler);
        return init.result.get();
    }

protected:

    virtual void asyncAddTask(const User& u, const ChunksData& c, const TaskInfo& t, const ContextPtr& ctx, onExecute cb) const = 0;
    virtual void asyncReadChunks(const User& u, const ChunkIds& chIds, const ContextPtr& ctx, onReadChunks cb) const = 0;
    virtual void asyncRemoveChunk(const User& u, const ChunkId& chId, const ContextPtr& ctx, onExecute cb) const = 0;
    virtual void asyncCanAddTask(const User& u, size_t tasksLimit, const ContextPtr& ctx, onBoolean cb) const = 0;
    virtual void asyncChooseChunkIds(const User& u, size_t limit, const ContextPtr& ctx, onChooseChunkIds cb) const = 0;
    virtual void asyncReadChunksByMids(const User& u, const Mids& mids, const ContextPtr& ctx, onReadChunks cb) const = 0;                            
    virtual void asyncChooseUsers(size_t limit, const ContextPtr& ctx, onChooseUsers cb) const = 0;                            
    virtual void asyncHasTasks(const User& u, const ContextPtr& ctx, onBoolean cb) const = 0;                            
    virtual void asyncUserStat(const User& u, const ContextPtr& ctx, onReadTasks cb) const = 0;                            
    virtual void asyncReadTask(const User& u, const TaskId& t, const ContextPtr& ctx, onReadTasks cb) const = 0;                            
    virtual void asyncAcquireLock(const User& u, const std::chrono::microseconds& timeout, const std::string& launchId, const ContextPtr& ctx, onBoolean cb) const = 0;                            
    virtual void asyncReleaseLock(const User& u, const std::string& launchId, const ContextPtr& ctx, onBoolean cb) const = 0;                            
};

using RepositoryPtr = std::shared_ptr<Repository>;

}//namespace ymod_taskmaster
