#include <mail/mops/ymod_taskmaster/ymod_db/include/repository.h>
#include <mail/mops/ymod_taskmaster/ymod_db/include/internal/query.h>
#include <mail/mops/ymod_taskmaster/ymod_db/include/internal/reflection.h>

#include <mail/mops/ymod_taskmaster/include/ymod_taskmaster/marshalling.hpp>
#include <mail/mops/ymod_taskmaster/include/internal/logger.hpp>

#include <mail/webmail/commondb/include/with_executers.h>
#include <mail/webmail/commondb/include/settings_reflection.h>
#include <mail/webmail/commondb/include/logger.h>

#include <yamail/data/deserialization/ptree.h>
#include <yamail/data/deserialization/json_reader.h>
#include <yamail/data/serialization/json_writer.h>

#include <yplatform/module_registration.h>

BOOST_FUSION_DEFINE_STRUCT((ymod_taskmaster), Settings,
    (std::string, reactor)
    (std::string, logger)
)

namespace ymod_taskmaster {

using namespace commondb;
using namespace std::string_literals;

inline boost::uuids::uuid convertUuid(const std::string& str) {
    return boost::lexical_cast<boost::uuids::uuid>(str);
}

namespace {

auto fromReflection(reflection::Chunk& chunk) {
    Chunk res;
    res.chunk.id = to_string(chunk.chunk_id);
    res.chunk.mids = std::move(chunk.mids);
    res.task = makeTaskInfo(chunk.task_info);
    return res;
}

auto fromReflection(reflection::BooleanResult& data) {
    return data.result;
}

auto fromReflection(reflection::ChooseChunkId& data) { 
    return to_string(data.chunk_id); 
};

auto fromReflection(reflection::User& user) { 
    return std::to_string(user.uid); 
};

auto fromReflection(reflection::TaskStatParams& params) {
    const auto info = makeTaskInfo(params.o_task_info);

    TaskStatParams result;
    result.setCreationTimestamp(params.o_created);
    result.setRemainedChunksCount(params.o_count);
    result.setTotalChunksCount(info.chunksCount);

    if (info.version == dataVersion) {
        result.setTaskId(info.id());
        result.setTaskGroupId(info.taskGroupId());
        result.setType(info.taskData.type);
    } else {
        result.setType(TaskType::Unknown);
    }
    return result;
}

} // namespase

struct Postgres: public WithExecuters, public Repository, public yplatform::module {

    virtual ~Postgres() = default;

    void init(const yplatform::ptree& cfg) {
        common = yamail::data::deserialization::fromPtree<Settings>(cfg);
        pg = readPgSettings(cfg.get_child("pg"));

        initExecuters(common.reactor, pg, query::QueriesRegister(), query::ParametersRegister());

        LOGDOG_(getModuleLogger(common.logger), notice,
            logdog::attr::message="ymod_taskmaster::Postgres module initialized",
            commondb::attr::settings=yamail::data::serialization::JsonWriter(common).result(),
            commondb::attr::pg_settings=yamail::data::serialization::JsonWriter(pg).result(),
            commondb::attr::password_length=passwordSize()
        );
    }
    
    auto master(const ContextPtr& ctx) const {
        auto ctxLogger = getContextLogger(ctx);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler("", ctx->request_id(), "mopsdb");
        return WithExecuters::readWriteExecutor(ctx->request_id(), logger, profiler);
    }

    auto anything(const ContextPtr& ctx) const {
        auto ctxLogger = getContextLogger(ctx);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler("", ctx->request_id(), "mopsdb");
        return WithExecuters::anyExecutor(ctx->request_id(), logger, profiler);
    }

    void asyncAddTask(const User& u, const ChunksData& c, const TaskInfo& t, const ContextPtr& ctx, onExecute cb) const override {
        master(ctx)->execute(
            pgg::withQuery<query::AddTask>(
                query::UserId(u.uid()),
                query::TaskId(convertUuid(t.id())),
                query::TaskInfo(toString(t)),
                query::ChunksInfo(toString(getChunksInfo(c)))
            ),
            std::move(cb)
        );
    }
    void asyncReadChunks(const User& u, const ChunkIds& chIds, const ContextPtr& ctx, onReadChunks cb) const override {
        std::vector<boost::uuids::uuid> ids;
        ids.reserve(chIds.size());
        std::transform(chIds.begin(), chIds.end(), std::back_inserter(ids), convertUuid);
        anything(ctx)->request(
            pgg::withQuery<query::GetChunks>(
                query::UserId(u.uid()),
                query::ChunkIds(std::move(ids))
            ),
            std::move(cb),
            [] (reflection::Chunk data) { return fromReflection(data); }
        );
    }
    void asyncRemoveChunk(const User& u, const ChunkId& chId, const ContextPtr& ctx, onExecute cb) const override {
        master(ctx)->execute(
            pgg::withQuery<query::RemoveChunk>(
                query::UserId(u.uid()),
                query::ChunkId(convertUuid(chId)),
                query::HostName(boost::asio::ip::host_name())
            ),
            std::move(cb)
        );
    }
    void asyncCanAddTask(const User& u, size_t tasksLimit, const ContextPtr& ctx, onBoolean cb) const override {
        anything(ctx)->request(
            pgg::withQuery<query::CanAddTask>(
                query::UserId(u.uid()), 
                query::Limit(tasksLimit)
            ),
            std::move(cb),
            [] (reflection::BooleanResult data) { return fromReflection(data); }
        );
    }
    void asyncChooseChunkIds(const User& u, size_t limit, const ContextPtr& ctx, onChooseChunkIds cb) const override {
        anything(ctx)->request(
            pgg::withQuery<query::ChooseChunkIds>(
                query::UserId(u.uid()),
                query::Limit(limit)
            ),
            std::move(cb),
            [] (reflection::ChooseChunkId data) { return fromReflection(data); }
        );
    }
    void asyncReadChunksByMids(const User& u, const Mids& mids, const ContextPtr& ctx, onReadChunks cb) const override {
        anything(ctx)->request(
            pgg::withQuery<query::GetChunksByMids>(
                query::UserId(u.uid()),
                query::Mids(mids)
            ),
            std::move(cb),
            [] (reflection::Chunk data) { return fromReflection(data); }
        );
    }
    void asyncChooseUsers(size_t limit, const ContextPtr& ctx, onChooseUsers cb) const override {
        anything(ctx)->request(
            pgg::withQuery<query::ChooseUsers>(
                query::Limit(limit)
            ),
            std::move(cb),
            [] (reflection::User data) { return fromReflection(data); }
        );
    }
    void asyncHasTasks(const User& u, const ContextPtr& ctx, onBoolean cb) const override {
        anything(ctx)->request(
            pgg::withQuery<query::HasTasks>(
                query::UserId(u.uid())
            ),
            std::move(cb),
            [] (reflection::BooleanResult data) { return fromReflection(data); }
        );
    }
    void asyncUserStat(const User& u, const ContextPtr& ctx, onReadTasks cb) const override {
        anything(ctx)->request(
            pgg::withQuery<query::UserStat>(
                query::UserId(u.uid())
            ),
            std::move(cb),
            [] (reflection::TaskStatParams data) { return fromReflection(data); }
        );
    }
    void asyncReadTask(const User& u, const TaskId& t, const ContextPtr& ctx, onReadTasks cb) const override {
        anything(ctx)->request(
            pgg::withQuery<query::GetTask>(
                query::UserId(u.uid()),
                query::TaskId(convertUuid(t))
            ),
            std::move(cb),
            [] (reflection::TaskStatParams data) { return fromReflection(data); }
        );
    }
    void asyncAcquireLock(const User& u, const std::chrono::microseconds& timeout, const std::string& launchId, const ContextPtr& ctx, onBoolean cb) const override {
        master(ctx)->request(
            pgg::withQuery<query::AcquireLock>(
                query::UserId(u.uid()),
                query::Timeout(timeout),
                query::LaunchId(launchId),
                query::HostName(boost::asio::ip::host_name())
            ),
            std::move(cb),
            [] (reflection::BooleanResult data) { return fromReflection(data); }
        );
    }
    void asyncReleaseLock(const User& u, const std::string& launchId, const ContextPtr& ctx, onBoolean cb) const override {
        master(ctx)->request(
            pgg::withQuery<query::ReleaseLock>(
                query::UserId(u.uid()),
                query::LaunchId(launchId)
            ),
            std::move(cb),
            [] (reflection::BooleanResult data) { return fromReflection(data); }
        );
    }

    Settings common;
    PgSettings pg;

};

}

DEFINE_SERVICE_OBJECT(ymod_taskmaster::Postgres)
