#pragma once

#include <db.h>
#include <boost/pointer_cast.hpp>
#include <boost/lexical_cast.hpp>
#include <multipaxos/types.h>
#include <multipaxos/acceptor_storage_interface.h>
#include <multipaxos/callback_logger.h>

/*
    Force Acceptors to write and flush to disk before answering
    to any request. This is required for fault-tolerance. The recovery procedure
    is not implemented at the moment.
    (0 = bdb auto, 1 = force disk sync)
*/
#define PAXOS_ACCEPTOR_FORCE_DISK_FLUSH 0

namespace multipaxos {

class acceptor_storage_bdb
    : public acceptor_storage_interface
    , public callback_logger
{
public:
    static const uint32_t STATE_RECORD_NO = 1;
    uint64_t MAX_RECORDS = 500000UL;

    acceptor_storage_bdb(
        const std::string& dbFileName,
        const unsigned max_records,
        bool removeOld = false,
        log_sev_function_t const& log_func = log_sev_function_t());

    ~acceptor_storage_bdb();

    virtual int force_syncrony()
    {
        STORAGE_LOG(debug, "Forcing disk synchrony");

        scoped_lock lock(mux_);
        return db_->sync(db_, 0);
    }

    virtual void update_state(const acceptor_state& state)
    {
        put_impl(
            STATE_RECORD_NO,
            state,
            sizeof(state.current_ballot) + sizeof(state.min_slot) + sizeof(state.max_slot));
    }

    virtual void update_record(const acceptor_record& rec)
    {
        if (rec.slot == -1)
        {
            throw std::domain_error("can't update record for special slot (-1)");
        }
        put_impl(
            rotatedSlotNumber(rec.slot),
            rec,
            sizeof(rec.slot) + sizeof(rec.ballot) + sizeof(rec.value.size()));
    }

    virtual std::unique_ptr<acceptor_state> lookup_state()
    {
        return get_impl<uint32_t, acceptor_state>(STATE_RECORD_NO);
    }

    // Return the most recent (based on ballot)
    // record info on disk

    virtual std::unique_ptr<acceptor_record> lookup_record(iid_t slot)
    {
        std::unique_ptr<acceptor_record> rec =
            get_impl<uint32_t, acceptor_record>(rotatedSlotNumber(slot));
        // reset if already collected
        if (rec.get() && rec->slot != slot)
        {
            rec.reset();
        }
        return rec;
    }

    virtual value_triplets lookup_records(const iid_t fromSlot, const iid_t toSlot)
    {
        value_triplets triplets;
        for (iid_t slot = fromSlot; slot <= toSlot; ++slot)
        {
            std::unique_ptr<acceptor_record> rec = lookup_record(slot);
            if (rec.get() && rec->slot == slot)
            {
                triplets.add(*rec);
            }
        }
        return triplets;
    }

private:
    template <typename KeyT, typename ValueT>
    void put_impl(const KeyT key, const ValueT& value, const size_t preReserveSize = 0)
    {
        DBT dbKey, dbData;
        memset(&dbKey, 0, sizeof(DBT));
        memset(&dbData, 0, sizeof(DBT));
        db_recno_t recno = key;
        dbKey.data = &recno;
        dbKey.size = sizeof(db_recno_t);
        msgpack::sbuffer sbuf(dbKey.size + preReserveSize);
        msgpack::pack(&sbuf, value);
        // data is flat_rec
        dbData.data = boost::static_pointer_cast<void>(sbuf.data());
        dbData.size = static_cast<uint32_t>(sbuf.size());

        scoped_lock lock(mux_);
        int ret = db_->put(db_, NULL, &dbKey, &dbData, 0);
        if (ret != 0)
        {
            STORAGE_LOG(error, "bdb PUT key:'" << key << "' failed: " << db_strerror(ret));
            throw std::runtime_error(
                "bdb can't put value for key " + boost::lexical_cast<string>(key));
        }
        // force disk flush
        if (PAXOS_ACCEPTOR_FORCE_DISK_FLUSH)
        {
            db_->sync(db_, 0);
        }
    }

    template <typename KeyT, typename ValueT>
    std::unique_ptr<ValueT> get_impl(const KeyT key)
    {
        int ret;
        DBT dbKey, dbData;

        memset(&dbKey, 0, sizeof(DBT));
        memset(&dbData, 0, sizeof(DBT));

        // Key is iid + 1
        db_recno_t recno = key;
        dbKey.data = &recno;
        dbKey.size = sizeof(db_recno_t);

        scoped_lock lock(mux_);

        ret = db_->get(db_, NULL, &dbKey, &dbData, 0);
        if (ret == DB_NOTFOUND || ret == DB_KEYEMPTY)
        {
            return std::unique_ptr<ValueT>();
        }

        if (ret != 0)
        {
            STORAGE_LOG(error, "lookup_record failed: " << db_strerror(ret));
            return std::unique_ptr<ValueT>();
        }

        try
        {
            std::unique_ptr<ValueT> rec(new ValueT());
            msgpack::unpacked unpacked_msg =
                msgpack::unpack(reinterpret_cast<char*>(dbData.data), dbData.size);
            msgpack::object obj = unpacked_msg.get();
            obj.convert(*rec.get());
            return rec;
        }
        catch (std::bad_cast& e)
        {
            STORAGE_LOG(error, "lookup_record unpack " << key << " failed: " << e.what());
            throw std::runtime_error(string("lookup_record unpackfailed: ") + e.what());
        }
    }

    uint32_t rotatedSlotNumber(iid_t slot)
    {
        return static_cast<uint32_t>((slot % MAX_RECORDS) + 2);
    }
    db_recno_t stateRecNumber();

    DB* db_; /* Database containing inventory information */

    mutex mux_;
    //    StorageStats stats_;
};

}
