#pragma once

#include <inttypes.h>
#include <vector>
#include <string.h>
#include <msgpack.hpp>
#include <memory>

namespace multipaxos {

typedef char byte_t;
typedef std::shared_ptr<byte_t> shared_carray;

inline shared_carray allocate_shared_carray(size_t size)
{
    return size ? shared_carray(new byte_t[size], std::default_delete<byte_t[]>()) :
                  shared_carray();
}

/*
 * Continuous shared buffer.
 */
class buffer_t
{
public:
    buffer_t() : offset_(0), size_(0)
    {
    }

    explicit buffer_t(shared_carray buffer, size_t size) : impl_(buffer), offset_(0), size_(size)
    {
    }

    explicit buffer_t(shared_carray buffer, const byte_t* data_begin, size_t size)
        : impl_(buffer), offset_(data_begin == nullptr ? 0 : data_begin - impl_.get()), size_(size)
    {
    }

    byte_t* data()
    {
        return impl_.get() + offset_;
    }

    const byte_t* data() const
    {
        return impl_.get() + offset_;
    }

    size_t size() const
    {
        return size_;
    }

    const shared_carray& native() const
    {
        return impl_;
    }

    static buffer_t allocate(size_t size)
    {
        buffer_t result;
        result.impl_ = allocate_shared_carray(size);
        result.offset_ = 0;
        result.size_ = size;
        return result;
    }

    template <typename Container>
    static buffer_t create_from(const Container& src)
    {
        buffer_t result = allocate(src.size());
        memcpy(result.data(), src.data(), src.size());
        return result;
    }

    operator bool() const
    {
        return size();
    }

    void reset()
    {
        impl_.reset();
        offset_ = 0;
        size_ = 0;
    }

    template <typename Packer>
    void msgpack_pack(Packer& packer) const
    {
        packer.pack_str(static_cast<uint32_t>(size()));
        packer.pack_str_body(data(), static_cast<uint32_t>(size()));
    }

    void msgpack_unpack(msgpack::object const& o)
    {
        if (o.type != msgpack::type::STR) throw msgpack::type_error();
        *this = buffer_t::allocate(o.via.str.size);
        memcpy(data(), o.via.str.ptr, o.via.str.size);
    }

    void msgpack_object(msgpack::object& o, msgpack::zone& /*z*/) const
    {
        o.type = msgpack::type::STR;
        o.via.str.ptr = data();
        o.via.str.size = static_cast<uint32_t>(size());
    }

    bool operator==(const buffer_t& other)
    {
        return size() == other.size() && data() == other.data();
    }

private:
    shared_carray impl_;
    size_t offset_;
    size_t size_;
};

inline bool buffer_equals(const buffer_t& buffer, const byte_t* val_data, size_t val_size)
{
    return buffer.size() == val_size && memcmp(buffer.data(), val_data, val_size) == 0;
}

inline bool buffer_equals(const buffer_t& buffer, const buffer_t& other)
{
    return buffer_equals(buffer, other.data(), other.size());
}

}
