#pragma once

#include <multipaxos/slots.h>
#include <sintimers/timers_queue_interface.h>

namespace multipaxos {

/**
 *
 * Frame - the circluar array of replicated log entries.
 * It's a not delivered part of the whole log.
 *
 * For example, the block marked with '*' is a frame of two proposed elements
 * on the continious abstract line of slots from 55 to 62.
 * |-----------*************-----------|
 * |   |   |   *     |     *   |   |   |
 * | C | C | C *  P  |  P  * - | - | - |
 * |   |   |   *     |     *   |   |   |
 * |-----------*************-----------|
 *   55  56  57   58   59    60  61  62
 *
 *  delivery = 55
 *  current == 58
 *  last_unused == 60
 */
struct frame_t
{
private:
    slot_n delivery; // first of slots ready to deliver
    slot_n current;  // the first uncommitted slot

    log_function_t* log_func;

    slot_n n_slots;
    boost::scoped_array<slot_t> slots; // replicated log data

public:
    void init(
        timers::queue_ptr timers_queue,
        slot_n initial_n,
        slot_n slots_count,
        log_function_t* plog_func = 0);

    slot_t& get_slot(slot_n n);

    const slot_t& get_slot(slot_n n) const;

    slot_n read_zone_begin() const;

    slot_n read_zone_end() const;

    slot_n read_zone_size() const;

    slot_n write_zone_begin() const;

    slot_n write_zone_end() const;

    slot_n write_zone_size() const;

    bool read_zone_has(slot_n n) const;

    bool write_zone_has(slot_n n) const;

    char state_char(const slot_t& slot) const;

    char state_char(slot_n n) const;

    void extend_read_zone(slot_n n);

    void cut_read_zone(slot_n n);

    slot_n preallocated_slots_count() const;
};
}