#pragma once

#include <multipaxos/types.h>
#include <multipaxos/paxos_config.h>
#include <multipaxos/messages.h>
#include <multipaxos/ntimer.h>
#include <multipaxos/answers.h>

#include <sintimers/timer.h>

#include <boost/accumulators/accumulators.hpp>
#include <boost/accumulators/statistics.hpp>

#include <set>

namespace acc = boost::accumulators;

namespace multipaxos {

typedef timers::timer_ptr timer_ptr;
typedef acc::accumulator_set<
    uint32_t,
    acc::stats<acc::tag::count, acc::tag::min, acc::tag::max, acc::tag::mean>>
    accumulator_t;

enum class state_t
{
    bad,
    idle,       // default (both)
    committed,  // committed (both)
    learn,      // learn (slave)
    wait,       // wait for proposing (master)
    propose,    // proposing now (master)
    accumulate, // accumulate while preparing (master)
};

// TODO log state changes
struct slot_t
{
public:
    slot_n num;
    ballot_t ballot;
    value_t value;

    accept_answers_t answers;
    timer_ptr timer;
    ntimer_t profile_timer;
    uint64_t commit_time;

    // public:
    slot_t();

    // access
    char state_char() const;
    state_t get_state() const;
    bool is_inited() const;
    bool is_inited_as(slot_n n) const;
    bool committed_as(slot_n n) const;

    // modify
    void init(slot_n n, ballot_t b, state_t s, value_t v);
    void reset();
    void reset_to_state(state_t new_state);

    void set_state(state_t new_state);
    bool add_answer(
        slot_n value_slot,
        ballot_t value_ballot,
        value_t value,
        acceptor_id_t const& acceptor_id);
    void reset_answers();
    void pull_answers_value();
    void cancel_timer();

private:
    state_t state;
};

struct prepare_state_t
{

    prepare_state_t() : ballot(-1), slot(-1), end_slot(-1), active(false), max_promised(-1)
    {
    }

    timer_ptr timer;
    ballot_t ballot;
    slot_n slot;
    slot_n end_slot;
    bool active;
    promise_answers_t answers;
    slot_n max_promised;

    void setup(ballot_t b, slot_n slot_begin, slot_n slot_end)
    {
        if (timer)
        {
            timer->cancel();
        }
        ballot = b;
        slot = slot_begin;
        end_slot = slot_end;
        active = true;
        answers.reset();
        max_promised = -1;
    }

    size_t get_answers_count()
    {
        return answers.get_count();
    }

    bool add_answer(promise_message const& msg, acceptor_id_t const& id)
    {
        if (msg.max_slot >= end_slot) max_promised = end_slot;
        return answers.insert(id);
    }

    void clear()
    {
        if (timer)
        {
            timer->cancel();
        }
        ballot = -1;
        slot = -1;
        active = false;
        answers.reset();
        max_promised = -1;
    }
};

#define N_SLOTS 131072 // replicated log circular length, the power of 2
//#define N_SLOTS 8

inline slot_n pow2roundup(slot_n x)
{
    --x;
    x |= x >> 1;
    x |= x >> 2;
    x |= x >> 4;
    x |= x >> 8;
    x |= x >> 16;
    return x + 1;
}

inline state_t slot_t::get_state() const
{
    assert(is_inited());
    return state;
}

inline bool slot_t::is_inited() const
{
    return num != -1;
}

inline bool slot_t::is_inited_as(slot_n n) const
{
    return num == n;
}

inline bool slot_t::committed_as(slot_n n) const
{
    return num == n && state == state_t::committed;
}

}
