#pragma once

#include <sintimers/queue.h>
#include <boost/asio/io_service.hpp>
#include <boost/asio/strand.hpp>

namespace multipaxos {

template <typename Logger>
class strand_async_ts_strategy
{
public:
    typedef timers::queue_ptr timers_queue_ptr;

private:
    typedef Logger logger_type;
    boost::asio::io_service ios;
    boost::asio::io_service::strand strand;
    logger_type logger;
    bool stopped;
    std::unique_ptr<boost::asio::io_service::work> work;
    timers_queue_ptr timers;
    boost::thread thread;

public:
    strand_async_ts_strategy(logger_type logger)
        : ios(1), strand(ios), logger(std::move(logger)), stopped(false)
    {
        timers = boost::make_shared<timers::queue_on_strand>(&strand);
    }

    void run()
    {
        work.reset(new boost::asio::io_service::work(ios));
        thread = boost::thread(boost::bind(&strand_async_ts_strategy::io_run_function, this));
    }

    void stop()
    {
        stopped = true;
        work.reset();
        ios.stop();
        thread.join();
    }

    timers_queue_ptr timers_queue()
    {
        return timers;
    }

    template <typename Callback>
    void call(Callback&& callback)
    {
        if (!stopped) strand.post(std::forward<Callback>(callback));
    }

private:
    void io_run_function()
    {
        while (!stopped)
        {
            try
            {
                ios.run();
                break;
            }
            catch (std::exception& e)
            {
                logger.error(
                    std::string("strand_async_ts_strategy::io_run_function exception: ") +
                    e.what());
            }
        }
    }
};

} // namespace multipaxos
