#pragma once

#include <multipaxos/acceptor.h>
#include <multipaxos/messages.h>

namespace multipaxos {

namespace detail {
template <typename Logger>
void log_func(Logger& logger, cb_severity_level severity, string const& s)
{
    switch (severity)
    {
    case cb_severity_level::error:
        logger.error(s);
        break;
    case cb_severity_level::info:
        logger.info(s);
        break;
    case cb_severity_level::debug:
        logger.debug(s);
        break;
    }
}
}

template <typename Network, typename Logger>
class ts_acceptor
{
    typedef Network network_type;
    typedef Logger logger_type;
    typedef typename network_type::address_type net_address_type;

public:
    ts_acceptor(
        std::shared_ptr<network_type> network,
        const logger_type& logger,
        std::shared_ptr<acceptor_storage_interface> storage)
        : network_(network), logger_(logger)
    {
        bind_network_messages();
        acceptor_ = std::make_shared<multipaxos::acceptor>(
            storage,
            network_->my_address(),
            boost::bind(detail::log_func<logger_type>, logger_, _1, _2));
    }

private:
    void bind_network_messages()
    {
        network_->template bind<prepare_message>(
            boost::bind(&ts_acceptor::on_prepare_message, this, _1, _2));
        network_->template bind<accept_message>(
            boost::bind(&ts_acceptor::on_accept_message, this, _1, _2));
        network_->template bind<sync_request_message>(
            boost::bind(&ts_acceptor::on_sync_message, this, _1, _2));
    }

    void on_prepare_message(const net_address_type& sender, const prepare_message& message)
    {
        acceptor_->prepare(
            message.ballot,
            message.slot,
            message.end_slot,
            boost::bind(&ts_acceptor::send_promise_message, this, _1, sender));
    }
    void on_accept_message(const net_address_type& sender, const accept_message& message)
    {
        acceptor_->accept(
            message.ballot,
            message.pvalue,
            boost::bind(&ts_acceptor::send_learn_message, this, _1),
            boost::bind(&ts_acceptor::send_reject_message, this, _1, sender));
    }
    void on_sync_message(const net_address_type& sender, const sync_request_message& message)
    {
        acceptor_->sync(
            message.slots, boost::bind(&ts_acceptor::send_sync_response_message, this, _1, sender));
    }

    void send_promise_message(
        const multipaxos::promise_message& message,
        const net_address_type& addr)
    {
        network_->send(message, addr);
    }

    void send_learn_message(const multipaxos::learn_message& message)
    {
        network_->broadcast(message);
    }

    void send_reject_message(
        const multipaxos::reject_message& message,
        const net_address_type& addr)
    {
        network_->send(message, addr);
    }

    void send_sync_response_message(
        const multipaxos::sync_response_message& message,
        const net_address_type& addr)
    {
        network_->send(message, addr);
    }

private:
    std::shared_ptr<network_type> network_;
    logger_type logger_;
    std::shared_ptr<multipaxos::acceptor> acceptor_;
};

}
