#pragma once

#include <multipaxos/detail/event_tracker.h>
#include <multipaxos/strand_async_ts_strategy.h>
#include <multipaxos/mp_sc_queue.h>
#include <multipaxos/types.h>
#include <multipaxos/basic_agent.h>
#include <sintimers/queue.h>
#include <boost/asio/strand.hpp>
#include <boost/lockfree/spsc_queue.hpp>

namespace multipaxos {

/**
 * Thread safe agent - wrapper for basic agent.
 */
template <
    typename Network,
    typename Logger,
    typename TSStrategy = strand_async_ts_strategy<Logger>,
    template <typename T> class Queue = mp_sc_queue>
class ts_agent
{
    typedef TSStrategy ts_strategy_type;
    typedef Queue<value_t> queue_type;
    typedef Network network_type;
    typedef Logger logger_type;
    typedef typename network_type::address_type net_address_type;

public:
    ts_agent()
    {
        stop_ = true;
        master_ = false;
        max_submits_per_iteration_ = 10;
        max_submits_queue_ = 10000;
    }

    ts_agent(const logger_type& logger) : logger(logger)
    {
        stop_ = true;
        master_ = false;
        max_submits_per_iteration_ = 10;
        max_submits_queue_ = 10000;
    }

    void init_network(std::shared_ptr<network_type> network)
    {
        network_ = network;
        network_->template bind<promise_message>(
            boost::bind(&ts_agent::on_promise_message, this, _1, _2));
        network_->template bind<learn_message>(
            boost::bind(&ts_agent::on_learn_message, this, _1, _2));
        network_->template bind<reject_message>(
            boost::bind(&ts_agent::on_reject_message, this, _1, _2));
        network_->template bind<sync_response_message>(
            boost::bind(&ts_agent::on_sync_response_message, this, _1, _2));
        network_->template bind<master_announce_message>(
            boost::bind(&ts_agent::on_master_announce_message, this, _1, _2));
    }

    void init(
        const algorithm_settings_t& algorithm_settings,
        unsigned node_id,
        unsigned total_log_size,
        iid_t start_from,
        const deliver_value_function_t& deliver_cb,
        const drop_function_t& drop_cb,
        const report_function_t& report_cb)
    {
        deliver_cb_ = deliver_cb;
        drop_cb_ = drop_cb;
        report_cb_ = report_cb;

        ts_strategy_.reset(new ts_strategy_type(logger));

        auto timers = ts_strategy_->timers_queue();

        multipaxos::settings_t log_settings;
        log_settings.algorithm_settings_t::operator=(algorithm_settings);
        log_settings.send_prepare_message = [this](const prepare_message& m) {
            network_->broadcast(m);
        };
        log_settings.send_accept_message = [this](const accept_message& m) {
            network_->broadcast(m);
        };
        log_settings.send_sync_message = [this](const sync_request_message& m) {
            network_->broadcast(m);
        };
        log_settings.send_announce_message = [this](const master_announce_message& m) {
            network_->broadcast(m);
        };
        log_settings.deliver_func = boost::bind(&ts_agent::deliver_func, this, _1);
        log_settings.drop_func = drop_cb_;
        if (logger.log_level() <= logger_type::level::info)
            log_settings.log_func = [this](const string& s) mutable { logger.info(s); };
        if (logger.log_level() <= logger_type::level::debug)
            log_settings.debug_log_func = [this](const string& s) mutable { logger.debug(s); };
        log_settings.report_func = boost::bind(&ts_agent::report_func, this, _1);

        d_log.reset(
            new multipaxos::basic_agent(log_settings, timers, start_from, total_log_size, node_id));

        d_current = start_from;

        master_ = false;

        submits_queue_ = std::make_shared<queue_type>(max_submits_queue_);
    }

    void set_max_submits_per_iteration(const unsigned value)
    {
        max_submits_per_iteration_ = value;
    }

    void start()
    {
        assert(network_);
        stop_ = false;
        pending_promises_ = 0;
        pending_rejects_ = 0;
        pending_learns_ = 0;
        pending_syncs_ = 0;
        ts_strategy_->call(boost::bind(&ts_agent::start_isolated, this));
        ts_strategy_->run();
    }

    void stop()
    {
        stop_ = true;
        ts_strategy_->call(boost::bind(&ts_agent::stop_isolated, this));

        ts_strategy_->stop();

        d_log.reset();
        ts_strategy_.reset();

        submits_queue_.reset();

        deliver_cb_.clear();
        drop_cb_.clear();
        report_cb_.clear();
    }

    bool submit(value_t value)
    {
        if (stop_) return false;
        if (!submits_queue_->push(value))
        {
            drop_cb_(value);
            return false;
        }
        bool need_handling = submits_tracker_.produce(1);
        if (need_handling) ts_strategy_->call(boost::bind(&ts_agent::submit_isolated, this));
        return true;
    }

    void set_master()
    {
        master_ = true;
        ts_strategy_->call(boost::bind(&ts_agent::update_mastership_isolated, this));
    }

    void set_not_master()
    {
        master_ = false;
        ts_strategy_->call(boost::bind(&ts_agent::update_mastership_isolated, this));
    }

    const stats_t& get_stats() const
    {
        return d_log->get_stats();
    }

private:
    void start_isolated()
    {
        /// XXX rename in common words
        d_log->start();
    }

    void stop_isolated()
    {
        d_log->stop();
        drop_all_submits_isolated();
    }

    void update_mastership_isolated()
    {
        if (stop_) return;
        bool last_value = master_;
        if (last_value != d_log->is_master())
            last_value ? d_log->set_master() : d_log->set_not_master();
        if (!last_value)
        {
            drop_all_submits_isolated();
        }
    }

    void submit_isolated()
    {
        if (stop_) return;
        unsigned count = 0;
        value_t value;
        while (submits_queue_->pop(value))
        {
            ++count;
            d_log->submit(value);
            if (max_submits_per_iteration_ && count >= max_submits_per_iteration_) break;
        }
        bool need_handling = submits_tracker_.consume(count);
        if (need_handling) ts_strategy_->call(boost::bind(&ts_agent::submit_isolated, this));
    }

    void drop_all_submits_isolated()
    {
        unsigned count = 0;
        value_t value;
        while (submits_queue_->pop(value))
        {
            ++count;
            drop_cb_(value);
        }
    }

    // TODO deliver all from basic_agent directly
    void deliver_func(iid_t /*n*/)
    {
        if (stop_) return;
        value_t val;
        multipaxos::slot_profile_t profile;
        while (get_status_t::ok == d_log->get(d_current, val, profile))
        {
            try
            {
                deliver_cb_(d_current, val, profile);
            }
            catch (const std::exception& e)
            {
                logger.error(string("ts_agent::deliver_func callback exception=") + e.what());
            }
            catch (...)
            {
                logger.error("ts_agent::deliver_func callback exception=unknown");
            }

            ++d_current;
        }
    }

    void report_func(report_code code)
    {
        if (code == report_code::prepare_timeout || code == report_code::accept_timeout)
        {
            logger.error(
                string("ts_agent::report timeout") + " pending_promises " +
                to_string(pending_promises_) + " pending_rejects " + to_string(pending_rejects_) +
                " pending_learns " + to_string(pending_learns_));
        }
        if (report_cb_) report_cb_(code);
    }

    void promise_received_isolated(promise_message const& message, const net_address_type& sender)
    {
        pending_promises_--;
        d_log->promise_received(message, sender);
    }

    void learn_received_isolated(learn_message const& message, const net_address_type& sender)
    {
        pending_learns_--;
        d_log->learn_received(message, sender);
    }

    void reject_received_isolated(reject_message const& message, const net_address_type& sender)
    {
        pending_rejects_--;
        d_log->reject_received(message, sender);
    }

    void sync_response_received_isolated(
        sync_response_message const& message,
        const net_address_type& sender)
    {
        pending_syncs_--;
        d_log->sync_response_received(message, sender);
    }

    void master_announce_received_isolated(
        master_announce_message const& message,
        const net_address_type& /*sender*/)
    {
        d_log->master_announce_received(message);
    }

    void on_promise_message(const net_address_type& sender, const promise_message& message)
    {
        if (stop_) return;
        pending_promises_++;
        ts_strategy_->call(
            boost::bind(&ts_agent::promise_received_isolated, this, message, sender));
    }

    void on_learn_message(const net_address_type& sender, const learn_message& message)
    {
        if (stop_) return;
        pending_learns_++;
        ts_strategy_->call(boost::bind(&ts_agent::learn_received_isolated, this, message, sender));
    }

    void on_reject_message(const net_address_type& sender, const reject_message& message)
    {
        if (stop_) return;
        pending_rejects_++;
        ts_strategy_->call(boost::bind(&ts_agent::reject_received_isolated, this, message, sender));
    }

    void on_sync_response_message(
        const net_address_type& sender,
        const sync_response_message& message)
    {
        if (stop_) return;
        pending_syncs_++;
        ts_strategy_->call(
            boost::bind(&ts_agent::sync_response_received_isolated, this, message, sender));
    }

    void on_master_announce_message(
        const net_address_type& sender,
        const master_announce_message& message)
    {
        if (stop_) return;
        ts_strategy_->call(
            boost::bind(&ts_agent::master_announce_received_isolated, this, message, sender));
    }

    std::shared_ptr<network_type> network_;
    logger_type logger;
    std::shared_ptr<ts_strategy_type> ts_strategy_;
    std::shared_ptr<basic_agent> d_log;
    iid_t d_current;
    std::atomic<bool> stop_;
    std::atomic<bool> master_;
    std::shared_ptr<queue_type> submits_queue_;
    detail::event_tracker submits_tracker_;
    unsigned max_submits_per_iteration_;
    unsigned max_submits_queue_;
    std::atomic<unsigned> pending_promises_;
    std::atomic<unsigned> pending_rejects_;
    std::atomic<unsigned> pending_learns_;
    std::atomic<unsigned> pending_syncs_;

    deliver_value_function_t deliver_cb_;
    drop_function_t drop_cb_;
    report_function_t report_cb_;
};

} // namespace multipaxos
