#include <multipaxos/acceptor.h>
#include <multipaxos/acceptor_storage_interface.h>

#define PAXOS_DEBUG 0

using namespace multipaxos;

acceptor::acceptor(
    std::shared_ptr<acceptor_storage_interface> storage,
    const acceptor_id_t& acceptor_id,
    log_sev_function_t const& log_func_)
    : callback_logger(log_func_), storage_(storage), acceptor_id_(acceptor_id)
{
    std::unique_ptr<acceptor_state> restoredState = storage_->lookup_state();
    if (restoredState.get())
    {
        state_ = *restoredState;
        ACCEPTOR_LOG(
            info, "restored state " << state_.current_ballot << " slot " << state_.max_slot);
    }
    else
    {
        ACCEPTOR_LOG(info, "no restored state");
    }
}

acceptor::~acceptor()
{
}

acceptor_state acceptor::state() const
{
    scoped_lock lock(mux_);
    return state_;
}

void acceptor::prepare(
    const ballot_t ballot,
    iid_t slot,
    const iid_t end_slot,
    const send_promise_callback& promise_callback)
{
    // @todo check min_slot <= slot <= max_slot (?)
    if (slot == -1)
    {
        return;
    }
    scoped_lock lock(mux_);
    if (ballot > state_.current_ballot)
    {
        ACCEPTOR_LOG(
            debug,
            "preparing next ballot " << ballot << ", current was " << state_.current_ballot
                                     << ", req-slot " << slot);
        updateBallot(ballot);
        promise_message response = { ballot,
                                     slot,
                                     state_.current_ballot,
                                     get_accepted_values(slot, end_slot),
                                     acceptor_id_,
                                     state_.max_slot };

        lock.unlock();
        // w/o lock
        promise_callback(response);
    }
    else
    {
        if (PAXOS_DEBUG)
        {
            ACCEPTOR_LOG(
                debug,
                "prepare for ballot " << ballot << " <= current " << state_.current_ballot
                                      << ", req-slot " << slot);
        }
        promise_message response = { ballot,           slot,         state_.current_ballot,
                                     value_triplets(), acceptor_id_, state_.max_slot };

        lock.unlock();
        // w/o lock
        promise_callback(response);
    }
}

void acceptor::accept(
    const ballot_t ballot,
    const slot_value_pair& pvalue,
    const send_learn_callback& learn_callback,
    const send_reject_callback& rejectCallback)
{
    if (pvalue.slot == -1)
    {
        return;
    }
    scoped_lock lock(mux_);
    if (ballot >= state_.current_ballot)
    {
        ACCEPTOR_LOG(
            debug,
            "accept for ballot " << ballot << ", current was " << state_.current_ballot
                                 << ", req-slot " << pvalue.slot);
        update_value(state_.current_ballot, pvalue); // write value before state updated
        update_state(ballot, pvalue.slot);
        learn_message response = { state_.current_ballot, pvalue, acceptor_id_ };

        lock.unlock();

        // w/o lock
        learn_callback(response);
    }
    else
    {
        if (PAXOS_DEBUG)
        {
            ACCEPTOR_LOG(
                debug,
                "rejecting accept for ballot " << ballot << " < current " << state_.current_ballot
                                               << ", req-slot " << pvalue.slot);
        }
        reject_message response = { ballot, pvalue.slot, state_.current_ballot, acceptor_id_ };

        lock.unlock();

        // w/o lock
        rejectCallback(response);
    }
}

void acceptor::sync(const std::set<iid_t>& slots, const send_sync_reponse_callback& cb)
{
    scoped_lock lock(mux_);
    sync_response_message response = { get_accepted_values(slots), state_.max_slot, acceptor_id_ };
    lock.unlock();
    // w/o lock
    cb(response);
}

void acceptor::updateBallot(const ballot_t ballot)
{
    if (state_.current_ballot < ballot)
    {
        state_.current_ballot = ballot;
        storage_->update_state(state_);
    }
}

void acceptor::update_state(const ballot_t ballot, const iid_t slot)
{
    bool state_changed = false;
    if (state_.current_ballot < ballot)
    {
        state_.current_ballot = ballot;
        state_changed = true;
    }
    if (state_.max_slot < slot)
    {
        state_.max_slot = slot;
        state_changed = true;
    }
    if (state_changed)
    {
        storage_->update_state(state_);
    }
}

value_triplets acceptor::get_accepted_values(const std::set<iid_t>& slots)
{
    value_triplets result;
    for (iid_t slot : slots)
    {
        if (slot <= state_.max_slot)
        {
            std::unique_ptr<acceptor_record> value = storage_->lookup_record(slot);
            if (value.get())
            {
                result.add(*value);
            }
        }
    }
    return result;
}

value_triplets acceptor::get_accepted_values(const iid_t slot, const iid_t end_slot)
{
    return storage_->lookup_records(slot, std::min(end_slot - 1, state_.max_slot));
}

void acceptor::update_value(const ballot_t ballot, const slot_value_pair& pvalue)
{
    if (pvalue.slot == -1)
    {
        throw std::domain_error("can't update value for reserved slot -1");
    }
    ballot_slot_value_triplet newItem;
    newItem.ballot = ballot;
    newItem.slot = pvalue.slot;
    newItem.value = pvalue.value;

    acceptor_record& rec = newItem; // no convertion needed while has the same type
    storage_->update_record(rec);
}
