#include <multipaxos/acceptor_storage_bdb.h>

/* BDB Access method, only 1 should be defined */
// #define BDB_USE_BTREE
// #define BDB_USE_RECNO
#define BDB_USE_HASH

namespace multipaxos {

acceptor_storage_bdb::acceptor_storage_bdb(
    const std::string& dbFileName,
    const unsigned max_records,
    bool removeOld,
    log_sev_function_t const& log_func)
    : callback_logger(log_func), MAX_RECORDS(max_records)
{
    db_ = NULL;

    if (removeOld)
    {
        // removes the previous DB if there
        remove(dbFileName.c_str());
    }

    /* Initialize the DB handle */
    int ret = db_create(&db_, NULL, 0);
    if (ret != 0)
    {
        STORAGE_LOG(error, "Create: " << dbFileName << ": " << db_strerror(ret));
        throw std::domain_error(
            "Acceptor database create failed: " + std::string(db_strerror(ret)));
    }

    /* Set up error handling for this database */
    // db_->set_errfile(db_, stdout);
    // db_->set_errpfx(db_, "program_name");

    DBTYPE accessMethod;
#ifdef BDB_USE_BTREE
    accessMethod = DB_BTREE;
#endif
#ifdef BDB_USE_RECNO
    accessMethod = DB_RECNO;
#endif
#ifdef BDB_USE_HASH
    accessMethod = DB_HASH;
#endif
    /* Set the open flags */
    uint32_t openFlags = DB_CREATE;

    /* Now open the database */
    ret = db_->open(
        db_,                /* Pointer to the database */
        NULL,               /* Txn pointer */
        dbFileName.c_str(), /* File name */
        NULL,               /* Logical db name (unneeded) */
        accessMethod,       /* Database type */
        openFlags,          /* Open flags */
        0);                 /* File mode. Using defaults */
    if (ret != 0)
    {
        db_->err(db_, ret, "Database '%s' open failed.", dbFileName.c_str());
        throw std::domain_error("Acceptor database open failed: " + std::string(db_strerror(ret)));
    }

    STORAGE_LOG(info, "Opened stable storage :'" << dbFileName << "'");

    /*
    {
        DB_BTREE_STAT *stats = NULL;
        stats = (DB_BTREE_STAT*) malloc(sizeof(DB_BTREE_STAT));
        if((ret = db_->stat(db_, NULL, &stats, 0)) != 0) {
            db_->err(db_, ret, "\nError collecting statistics");
        }
        printf("Key count = %d, Data count = %d\n", stats->bt_nkeys, stats->bt_ndata);
        STORAGE_LOG(info, "DB '" << dbFileName
            << "' opened, Key count = " << stats->bt_nkeys
            << ", Data count = " << stats->bt_ndata);

        DBC *cursorp;
        DBT key, data;
        int ret;

        // Initialize cursor
        db_->cursor(db_, NULL, &cursorp, 0);

        // Initialize our DBTs.
        memset(&key, 0, sizeof(DBT));
        memset(&data, 0, sizeof(DBT));

        // Iterate over the database, retrieving each record in turn.
        while (!(ret=cursorp->c_get(cursorp, &key, &data, DB_NEXT))) {
            uint32_t nkey = *(uint32_t*)key.data;
            printf("%d - %s\n", nkey);
        };
        if (ret != DB_NOTFOUND) {
            printf("Nothing found in the database.\n");
        };

        // Close cursor before exit
        if (cursorp != NULL)
            cursorp->c_close(cursorp);
    }
    */
}

acceptor_storage_bdb::~acceptor_storage_bdb()
{
    int ret = db_->close(db_, 0);
    if (ret != 0)
    {
        STORAGE_LOG(error, "Database close failed: " << db_strerror(ret));
    }
}

}
