#include <multipaxos/basic_agent.h>

#include "logger.h"
#include "master_impl.h"
#include "slave_impl.h"
#include "common_methods.h"

#include <boost/noncopyable.hpp>
#include <boost/make_shared.hpp>

namespace multipaxos {

#ifdef LOG_ENABLED
#undef LOG_ENABLED
#endif
#define LOG_ENABLED settings_.log_func
#define L_STREAM                                                                                   \
    if (LOG_ENABLED) logger_t(settings_.log_func)
#define DEBUG_LOG_ENABLED settings.debug_log_func
#define L_DEBUG                                                                                    \
    if (DEBUG_LOG_ENABLED) logger_t(settings.debug_log_func)

basic_agent::basic_agent(
    const settings_t& settings,
    timers::queue_ptr timers_queue,
    slot_n initial_n,
    slot_n slots_count,
    unsigned node_id)
{
    node_id = node_id % MAX_PROPOSERS;
    settings_ = settings;

    if (!timers_queue)
    {
        L_STREAM << "WARNING no timers";
    }

    master = false;
    stopped_ = true;

    frame_.init(timers_queue, initial_n, slots_count, &settings_.log_func);
    master_impl_ = boost::make_shared<master_impl>(frame_, node_id, settings, stats_, timers_queue);
    slave_impl_ = boost::make_shared<slave_impl>(frame_, settings, timers_queue);
}

basic_agent::~basic_agent()
{
}

bool basic_agent::stopped() const
{
    return stopped_;
}

bool basic_agent::is_master() const
{
    return master;
}

void basic_agent::start()
{
    if (!stopped_) return;
    stopped_ = false;
    if (master) master_impl_->activate();
    else
        slave_impl_->activate();
}

void basic_agent::stop()
{
    if (stopped_) return;
    if (master) master_impl_->deactivate();
    else
        slave_impl_->deactivate();
    stopped_ = true;
}

const stats_t& basic_agent::get_stats() const
{
    return stats_;
}

void basic_agent::drop_value(value_t value)
{
    if (settings_.drop_func)
    {
        assert(value);
        settings_.drop_func(value);
    }
}

void basic_agent::submit(value_t value)
{
    if (stopped_)
    {
        L_STREAM << "submit canceled because of stopped log";
        drop_value(value);
        return;
    }

    if (master)
    {
        stats_.submits++;
        master_impl_->submit(value);
    }
    else
    {
        L_STREAM << "submit canceled on slave";
        drop_value(value);
        return;
    }
}

get_status_t basic_agent::get(slot_n num, value_t& value, slot_profile_t& profile)
{
    if (master) return master_impl_->get(num, value, profile);
    else
        return ::multipaxos::get(frame_, settings_.debug_log_func, num, value, profile);
}

void basic_agent::learn_received(learn_message const& msg, acceptor_id_t const& acceptor_id)
{
    if (stopped_) return;
    stats_.learns_received++;
    auto from = msg.acceptor_id != "" ? msg.acceptor_id : acceptor_id;
    if (master)
    {
        master_impl_->learn_received(
            msg.accepted_value.slot, msg.ballot, msg.accepted_value.value, from);
    }
    else
    {
        slave_impl_->pick_value(
            msg.accepted_value.slot, msg.ballot, msg.accepted_value.value, from);
    }
}

void basic_agent::promise_received(promise_message const& msg, acceptor_id_t const& acceptor_id)
{
    if (stopped_) return;
    if (!master) return;

    auto from = msg.acceptor_id != "" ? msg.acceptor_id : acceptor_id;
    master_impl_->promise_received(msg, from);
}

void basic_agent::reject_received(reject_message const& msg, acceptor_id_t const& acceptor_id)
{
    if (stopped_) return;
    if (!master) return;

    auto from = msg.acceptor_id != "" ? msg.acceptor_id : acceptor_id;
    master_impl_->reject_received(msg, acceptor_id);
}

void basic_agent::sync_response_received(
    sync_response_message const& msg,
    acceptor_id_t const& acceptor_id)
{
    if (stopped_) return;
    if (master) return;

    auto from = msg.acceptor_id != "" ? msg.acceptor_id : acceptor_id;
    slave_impl_->sync_received(msg, acceptor_id);
}

void basic_agent::master_announce_received(master_announce_message const& msg)
{
    if (stopped_) return;
    if (master) return;

    slave_impl_->master_announce_received(msg);
}

void basic_agent::set_master()
{
    if (stopped_) return;
    if (!master)
    {
        L_STREAM << "op=setmaster"
                 << " max_parallel_accepts=" << settings_.max_parallel_accepts
                 << " max_prepare_step=" << settings_.max_prepare_step
                 << " drop_submits_while_preparing=" << settings_.drop_submits_while_preparing;
        master = true;
        slave_impl_->deactivate();
        master_impl_->activate();
    }
}

void basic_agent::set_not_master()
{
    if (stopped_) return;
    if (master)
    {
        L_STREAM << "op=setslave"
                 << " sync_max_size=" << settings_.sync_max_size
                 << " lag_to_report=" << settings_.lag_to_report;
        master = false;
        master_impl_->deactivate();
        slave_impl_->activate();
    }
}

}
