#include <multipaxos/frame.h>

namespace multipaxos {

void frame_t::init(
    timers::queue_ptr timers_queue,
    slot_n initial_n,
    slot_n slots_count,
    log_function_t* plog_func)
{
    log_func = plog_func;
    n_slots = pow2roundup(slots_count);
    slots.reset(new slot_t[n_slots]);

    for (slot_n i = 0; i < n_slots; ++i)
    {
        slots[i].reset();
        if (timers_queue)
        {
            slots[i].timer = timers_queue->create_timer();
        }
    }

    delivery = initial_n;
    current = initial_n;
}

slot_t& frame_t::get_slot(slot_n n)
{
    return slots[n & (n_slots - 1)];
}

const slot_t& frame_t::get_slot(slot_n n) const
{
    return slots[n & (n_slots - 1)];
}

slot_n frame_t::read_zone_begin() const
{
    return delivery;
}

slot_n frame_t::read_zone_end() const
{
    return current;
}

slot_n frame_t::read_zone_size() const
{
    return read_zone_end() - read_zone_begin();
}

slot_n frame_t::write_zone_begin() const
{
    return current;
}

slot_n frame_t::write_zone_end() const
{
    return current + write_zone_size();
}

slot_n frame_t::write_zone_size() const
{
    return n_slots - read_zone_size();
}

bool frame_t::read_zone_has(slot_n n) const
{
    return n >= read_zone_begin() && n < read_zone_end();
}

bool frame_t::write_zone_has(slot_n n) const
{
    return n >= write_zone_begin() && n < write_zone_end();
}

char frame_t::state_char(const slot_t& slot) const
{
    return slot.state_char();
}

char frame_t::state_char(slot_n n) const
{
    return state_char(get_slot(n));
}

void frame_t::extend_read_zone(slot_n n)
{
    assert(write_zone_has(n) || n == write_zone_end());
    if (n == write_zone_end())
    {
        for (auto i = write_zone_begin(); i < n; ++i)
            assert(get_slot(i).committed_as(i));
    }
    current = n;
}

void frame_t::cut_read_zone(slot_n n)
{
    assert(read_zone_has(n) || n == read_zone_end());
    delivery = n;
}

slot_n frame_t::preallocated_slots_count() const
{
    return n_slots;
}

}
