#include "master_impl.h"

#include "common_methods.h"
#include "logger.h"

#define LOG_ENABLED settings.log_func
#define L_STREAM                                                                                   \
    if (LOG_ENABLED) logger_t(settings.log_func) << "master_log "
#define DEBUG_LOG_ENABLED settings.debug_log_func
#define L_DEBUG                                                                                    \
    if (DEBUG_LOG_ENABLED) logger_t(settings.debug_log_func) << "master_log "

#define FRAME_TO_STREAM                                                                            \
    " read=" << frame_.read_zone_begin() << " write=" << frame_.write_zone_begin()                 \
             << " window_end=" << window_end << " max_promised=" << max_promised

namespace multipaxos {

namespace {

value_t pick_answers(slot_t& slot)
{
    value_t val_to_resubmit;
    if (slot.value && !buffer_equals(slot.value, slot.answers.value))
    {
        val_to_resubmit = slot.value;
        slot.value = slot.answers.value;
    }
    else
    {
        slot.value = slot.answers.value;
    }
    slot.answers.reset();
    return val_to_resubmit;
}

}

master_impl::master_impl(
    frame_t& frame,
    const unsigned proposer_id,
    const settings_t& settings,
    stats_t& stats,
    timers::queue_ptr timers_queue)
    : frame_(frame), stats_(stats), settings(settings)
{
    active_ = false;
    prepared = false;
    // algorithm optimization: ballot offset for different proposers
    ballot = proposer_id;
    window_end = -1;
    max_promised = -1;
    if (timers_queue)
    {
        prepare_state.timer = timers_queue->create_timer();
        announce_timer = timers_queue->create_timer();
    }
}

bool master_impl::is_active() const
{
    return active_;
}

bool master_impl::is_prepared() const
{
    return prepared;
}

bool master_impl::is_my_ballot(ballot_t other) const
{
    return ballot == other;
}

void master_impl::activate()
{
    assert(!active_);
    active_ = true;
    window_end = frame_.write_zone_begin();
    start_prepare(ballot + 1);
    start_master_announce();
}

void master_impl::deactivate()
{
    assert(active_);

    cancel_all_async_operations();
    drop_proposals();
    active_ = false;
    prepared = false;
}

// move code up
void master_impl::submit(value_t value)
{
    assert(active_);
    assert(value);

    if (settings.drop_submits_while_preparing && !is_prepared())
    {
        drop_value(value);
        return;
    }

    if (!insert_proposal(value))
    {
        drop_value(value);
        return;
    }

    if (is_prepared())
    {
        propose_all_waiting_values();
    }
}

void master_impl::reject_received(reject_message const& msg, acceptor_id_t const& acceptor_id)
{
    assert(active_);
    if (is_prepared() && is_my_ballot(msg.request_ballot))
    {
        L_STREAM << "reject_received from " << acceptor_id << FRAME_TO_STREAM;
        reset_prepare(msg.acceptor_ballot + 1);
    }
}

get_status_t master_impl::get(slot_n num, value_t& value, slot_profile_t& profile)
{
    bool waiting_for_prepare = (!is_prepared() && frame_.write_zone_size() == 0);
    auto result = ::multipaxos::get(frame_, settings.debug_log_func, num, value, profile);
    if (result == get_status_t::ok && waiting_for_prepare && frame_.write_zone_size())
    {
        start_prepare(ballot);
    }
    return result;
}

void master_impl::promise_received(promise_message const& msg, acceptor_id_t const& acceptor_id)
{
    assert(active_);
    if (is_prepared()) return;
    if (!is_my_ballot(msg.requested_ballot)) return;
    if (!is_my_ballot(msg.acceptor_ballot))
    {
        reset_prepare(msg.acceptor_ballot + 1);
        return;
    }

    bool answer_inserted = prepare_state.add_answer(msg, acceptor_id);
    size_t answers_count = prepare_state.get_answers_count();
    bool majority = answers_count >= MAJORITY;
    L_DEBUG << "promise_received"
            << " inserted=" << (answer_inserted ? "yes" : "no")
            << " majority=" << (majority ? "yes" : "no") << " answers.size=" << answers_count
            << " values.size=" << msg.accepted_values.items().size()
            << " requested_slot=" << msg.requested_slot << " ballot=" << msg.requested_ballot
            << " from=" << acceptor_id << FRAME_TO_STREAM;

    if (answer_inserted)
    {
        for (const auto& item : msg.accepted_values.items())
        {
            if (!frame_.write_zone_has(item.slot))
            {
                continue;
            }
            auto& slot = frame_.get_slot(item.slot);
            if (slot.committed_as(item.slot)) continue;
            slot_accumulate(slot, item.slot, item.ballot, item.value, acceptor_id);
        }

        if (majority)
        {
            complete_prepare();
        }
    }
}

bool master_impl::learn_received(
    slot_n value_slot,
    ballot_t value_ballot,
    value_t value,
    const acceptor_id_t& acceptor_id)
{
    assert(active_);
    if (!is_prepared()) return false;
    if (!is_my_ballot(value_ballot))
    {
        stats_.learns_skipped_wrong_ballot++;
        return false;
    }
    if (!frame_.write_zone_has(value_slot))
    {
        return false;
    }

    if (!slot_learn(value_slot, value_ballot, value, acceptor_id)) return false;

    if (!promised(frame_.write_zone_begin()))
    {
        reset_prepare(ballot + 1);
    }
    else
    {
        propose_all_waiting_values();
    }

    return true;
}

void master_impl::drop_value(value_t value)
{
    if (settings.drop_func)
    {
        assert(value);
        settings.drop_func(value);
    }
}

bool master_impl::insert_proposal(value_t value)
{
    assert(window_end >= frame_.write_zone_begin());
    // TODO start from frame_.write_zone_begin()?
    for (slot_n i = window_end; i < frame_.write_zone_end(); ++i)
    {
        auto& slot = frame_.get_slot(i);
        if (!slot.is_inited_as(i) || slot.get_state() == state_t::learn ||
            slot.get_state() == state_t::bad || slot.get_state() == state_t::idle)
        {
            slot_insert(slot, i, value);
            return true;
        }
    }

    L_DEBUG << "j_fail insert_no_free_slot" << FRAME_TO_STREAM;
    return false;
}

bool master_impl::promised(slot_n slot)
{
    return max_promised == -1 || slot < max_promised;
}

void master_impl::propose_one(slot_t& slot)
{
    slot_propose(slot);

    accept_message msg;
    msg.ballot = ballot;
    msg.pvalue.slot = slot.num;
    msg.pvalue.value = slot.value;

    // TODO hide timer set and cancel in slot_t
    if (slot.timer)
    {
        slot.timer->async_wait(
            settings.propose_max_time, boost::bind(&master_impl::timeout, this, slot.num));
    }

    settings.send_accept_message(msg);
}

void master_impl::propose_all_waiting_values()
{
    assert(prepared);

    // TODO stop on first not waiting and not committed
    for (slot_n cur = window_end; cur < frame_.write_zone_end(); ++cur)
    {

        // parallel proposes limit exceeded
        if (cur - frame_.write_zone_begin() >= settings.max_parallel_accepts)
        {
            break;
        }

        if (!promised(cur))
        {
            break;
        }

        slot_t& slot = frame_.get_slot(cur);
        if (!slot.is_inited_as(cur) || slot.get_state() == state_t::idle)
        {
            break;
        }

        if (slot.get_state() == state_t::committed)
        {
            window_end++;
            continue;
        }

        if (slot.get_state() == state_t::wait)
        {
            propose_one(slot);
            window_end++;
        }
        else
        {
            L_STREAM << "WARNING propose_all_waiting_values_break slot=" << slot.num
                     << " ballot=" << ballot << " state=" << slot.state_char() << FRAME_TO_STREAM
                     << " iter_index=" << cur;
            break;
        }
    }
}

void master_impl::timeout(slot_n n)
{
    if (!active_ || !is_prepared()) return;

    // TODO do not reset prepare state, just re-send accept message
    slot_t& slot = frame_.get_slot(n);
    if (!slot.is_inited_as(n)) return;
    if (slot.get_state() == state_t::propose && is_my_ballot(slot.ballot))
    {
        L_STREAM << "accept_timeout timeout_slot=" << n << " my_ballot=" << ballot
                 << " slot=" << slot.num << " state=" << slot.state_char() << FRAME_TO_STREAM;
        if (settings.report_func)
        {
            settings.report_func(report_code::accept_timeout);
        }
        reset_prepare(ballot + 1);
    }
}

void master_impl::prepare_timeout()
{
    if (!active_ || is_prepared())
    {
        L_STREAM << "WARNING wrong prepare_timeout";
        return;
    }

    L_STREAM << "prepare_timeout";

    if (settings.report_func)
    {
        settings.report_func(report_code::prepare_timeout);
    }

    reset_prepare(ballot + 1);
}

void master_impl::reset_prepare(ballot_t minimal_ballot)
{
    if (is_prepared()) cancel_propose_operations();
    prepare_state.clear();
    prepared = false;
    max_promised = -1;
    stats_.reset_prepare++;
    start_prepare(minimal_ballot);
}

void master_impl::increase_ballot(ballot_t minimal_ballot)
{
    while (ballot < minimal_ballot)
    {
        ballot += MAX_PROPOSERS;
    }
}

void master_impl::start_prepare(ballot_t minimal_ballot)
{
    assert(!is_prepared());

    if (frame_.write_zone_size() == 0)
    {
        assert(frame_.read_zone_size());
        return;
    }

    increase_ballot(minimal_ballot);

    L_STREAM << "start_prepare ballot=" << ballot << FRAME_TO_STREAM;

    slot_n prepare_step =
        std::min(frame_.write_zone_size(), static_cast<slot_n>(settings.max_prepare_step));
    slot_n prepare_end = frame_.write_zone_begin() + prepare_step;

    prepare_state.setup(ballot, frame_.write_zone_begin(), prepare_end);

    if (prepare_state.timer)
    {
        prepare_state.timer->async_wait(
            settings.prepare_max_time, boost::bind(&master_impl::prepare_timeout, this));
    }

    prepare_message msg;
    msg.ballot = ballot;
    msg.slot = frame_.write_zone_begin();
    msg.end_slot = prepare_end;

    settings.send_prepare_message(msg);
}

void master_impl::complete_prepare()
{
    max_promised = prepare_state.max_promised;
    prepare_state.clear();
    pull_accumulated_values();
    prepared = true;
    if (window_end < frame_.write_zone_begin()) window_end = frame_.write_zone_begin();

    L_STREAM << "complete_prepare ballot=" << ballot << " max_promised=" << max_promised
             << FRAME_TO_STREAM;

    if (!promised(frame_.write_zone_begin()))
    {
        reset_prepare(ballot + 1);
    }
    else
    {
        propose_all_waiting_values();
    }
}

void master_impl::pull_accumulated_values()
{
    for (slot_n i = frame_.write_zone_begin(); i < frame_.write_zone_end(); ++i)
    {
        slot_t& slot = frame_.get_slot(i);
        if (!slot.is_inited_as(i)) continue;
        if (slot.get_state() == state_t::accumulate)
        {
            assert(slot.answers.get_count() < MAJORITY);

            value_t val_to_resubmit;
            if (slot.value && !buffer_equals(slot.value, slot.answers.value))
            {
                val_to_resubmit = slot.value;
            }

            slot.init(slot.num, slot.answers.ballot, state_t::wait, slot.answers.value);

            if (val_to_resubmit)
            {
                L_STREAM << "pull_accumulated_values_resubmit slot=" << slot.num;
                if (!insert_proposal(val_to_resubmit))
                {
                    drop_value(val_to_resubmit);
                }
            }
        }
    }
}

void master_impl::cancel_all_async_operations()
{
    ntimer_t profile_timer;
    L_STREAM << "cancel_all_async_operations" << FRAME_TO_STREAM;
    for (slot_n i = frame_.write_zone_begin(); i < frame_.write_zone_end(); ++i)
    {
        slot_t& slot = frame_.get_slot(i);
        if (slot.timer) slot.timer->cancel();
    }
    if (prepare_state.timer)
    {
        prepare_state.timer->cancel();
    }
    stop_master_announce();
    L_STREAM << "cancel_all_async_operations finished time=" << to_string(profile_timer)
             << FRAME_TO_STREAM;
}

void master_impl::drop_proposals()
{
    ntimer_t profile_timer;
    L_STREAM << "drop_proposals" << FRAME_TO_STREAM;

    // TODO reserve at least for size of the frame window values
    std::vector<value_t> values_to_drop;

    for (slot_n i = frame_.write_zone_begin(); i < frame_.write_zone_end(); ++i)
    {
        slot_t& slot = frame_.get_slot(i);
        if (!slot.is_inited_as(i)) continue;
        if (slot.get_state() == state_t::propose || slot.get_state() == state_t::wait)
        {
            assert(slot.is_inited_as(i));
            if (slot.value)
            {
                values_to_drop.push_back(slot.value);
            }
            slot.reset();
        }
        else if (slot.get_state() == state_t::accumulate)
        {
            assert(slot.is_inited_as(i));
            slot.reset();
        }
    }

    window_end = frame_.write_zone_begin();

    L_STREAM << "drop_proposals finished time=" << to_string(profile_timer) << FRAME_TO_STREAM;

    for (auto i = values_to_drop.begin(); i != values_to_drop.end(); ++i)
    {
        drop_value(*i);
    }
}

void master_impl::start_master_announce()
{
    if (announce_timer && settings.send_announce_message)
    {
        announce_timer->async_wait(
            settings.announce_interval, boost::bind(&master_impl::master_announce_timeout, this));
    }
}

void master_impl::stop_master_announce()
{
    if (announce_timer)
    {
        announce_timer->cancel();
    }
}

void master_impl::master_announce_timeout()
{
    if (!active_ || !settings.send_announce_message) return;

    master_announce_message msg;
    msg.read = frame_.read_zone_begin();
    msg.write = frame_.write_zone_begin();
    settings.send_announce_message(msg);

    start_master_announce();
}

void master_impl::cancel_propose_operations()
{
    L_STREAM << "cancel_propose_operations" << FRAME_TO_STREAM;

    // TODO check slot states and log warnings
    for (slot_n i = frame_.write_zone_begin(); i < window_end; ++i)
    {
        slot_t& slot = frame_.get_slot(i);
        if (slot.is_inited_as(i) && slot.get_state() == state_t::propose)
        {
            slot_cancel_propose(slot);
        }
    }
    window_end = frame_.write_zone_begin();
}

bool master_impl::is_committed(slot_n n)
{
    slot_t& slot = frame_.get_slot(n);
    return slot.is_inited_as(n) && slot.get_state() == state_t::committed;
}

bool master_impl::shift_delivery_frame()
{
    slot_n committed_range_end = frame_.write_zone_begin();
    for (; committed_range_end < frame_.write_zone_end(); ++committed_range_end)
    {
        if (!is_committed(committed_range_end))
        {
            break;
        }
    }

    bool need_notify = (frame_.write_zone_begin() != committed_range_end);
    frame_.extend_read_zone(committed_range_end);
    // window_end must stay in write zone when accumulating
    // values with commits
    if (window_end < frame_.write_zone_begin())
    {
        window_end = frame_.write_zone_begin();
    }
    return need_notify;
}

void master_impl::slot_insert(slot_t& slot, slot_n n, value_t value)
{
    slot.init(n, ballot_t(-1), state_t::wait, value);
    L_DEBUG << "j_insert num=" << n << " value_sz=" << value.size();
}

void master_impl::slot_accumulate(
    slot_t& slot,
    slot_n n,
    ballot_t ballot,
    value_t value,
    const acceptor_id_t& acceptor_id)
{
    if (slot.is_inited_as(n)) slot.set_state(state_t::accumulate);
    else
        slot.init(n, -1, state_t::accumulate, value_t());

    bool majority = slot.add_answer(n, ballot, value, acceptor_id);
    if (majority)
    {
        auto pushed_value = pick_answers(slot);
        slot.set_state(state_t::committed);
        L_STREAM << "j_accumulate_commit num=" << n << " ballot=" << ballot
                 << " acceptor=" << acceptor_id << " value_sz=" << value.size() << FRAME_TO_STREAM;

        bool need_notify = shift_delivery_frame();
        if (need_notify && settings.deliver_func) settings.deliver_func(frame_.read_zone_end());

        if (pushed_value)
        {
            L_STREAM << "j_accumulate_move num=" << n << " ballot=" << ballot;
            insert_proposal(pushed_value);
        }
    }
    else
    {
        L_DEBUG << "j_accumulate num=" << n << " ballot=" << ballot << " acceptor=" << acceptor_id
                << " value_sz=" << value.size() << FRAME_TO_STREAM;
    }
}

bool master_impl::slot_learn(
    slot_n n,
    ballot_t ballot,
    value_t value,
    const acceptor_id_t& acceptor_id)
{
    slot_t& slot = frame_.get_slot(n);
    if (!slot.is_inited_as(n)) return false;
    if (slot.get_state() != state_t::propose) return false;

    bool majority = slot.add_answer(n, ballot, value, acceptor_id);
    if (majority)
    {
        slot.commit_time = slot.profile_timer.shot();
        if (slot.timer) slot.timer->cancel();

        std::stringstream ss;
        ss << " time=" << std::fixed << std::setprecision(3)
           << static_cast<double>(slot.commit_time) * 0.001;
        L_STREAM << "j_learn_commit num=" << n << " ballot=" << ballot
                 << " acceptor=" << acceptor_id << " value_sz=" << value.size() << ss.str()
                 << FRAME_TO_STREAM;

        auto pushed_value = pick_answers(slot);
        slot.set_state(state_t::committed);

        bool need_notify = shift_delivery_frame();
        if (need_notify && settings.deliver_func) settings.deliver_func(frame_.read_zone_end());

        if (pushed_value)
        {
            L_STREAM << "j_learn_move num=" << n << " ballot=" << ballot;
            insert_proposal(pushed_value);
        }
    }
    else
    {
        L_DEBUG << "j_learn num=" << n << " ballot=" << ballot << " acceptor=" << acceptor_id
                << " value_sz=" << value.size() << FRAME_TO_STREAM;
    }

    return true;
}

void master_impl::slot_propose(slot_t& slot)
{
    assert(slot.value);
    assert(slot.is_inited());
    slot.reset_answers();
    slot.ballot = ballot;
    slot.set_state(state_t::propose);
    slot.profile_timer.reset();
    L_DEBUG << "j_propose num=" << slot.num << " ballot=" << ballot << FRAME_TO_STREAM;
}

void master_impl::slot_cancel_propose(slot_t& slot)
{
    assert(slot.value);
    assert(slot.get_state() == state_t::propose);
    slot.init(slot.num, slot.ballot, state_t::wait, slot.value);
    L_DEBUG << "j_cancel_propose num=" << slot.num << " ballot=" << ballot << FRAME_TO_STREAM;
}

}
