#pragma once

#include <multipaxos/types.h>
#include <multipaxos/paxos_config.h>
#include <multipaxos/settings.h>
#include <multipaxos/frame.h>
#include <multipaxos/slots.h>
#include <multipaxos/messages.h>
#include <multipaxos/ntimer.h>
#include <multipaxos/timers_queue.h>
#include <multipaxos/stats.h>

namespace multipaxos {

/**
 * Master - smart proposer + learner in the Paxos terms.
 *
 * Proposing of values splits into 2 phases:
 * 1) prepare
 * 2) propose
 * In the Multi-Paxos algorithm phase 1 is made once for many values until
 * timeout happens or reject receinved.
 * So "prepared" - a special master's state.
 *
 * Master works with slots via a frame object.
 */
class master_impl
{
public:
    master_impl(
        frame_t& frame,
        const unsigned proposer_id,
        const settings_t& settings,
        stats_t& stats,
        timers::queue_ptr timers_queue);

    bool is_active() const;
    bool is_prepared() const;
    bool is_my_ballot(ballot_t ballot) const;

    void activate();
    void deactivate();
    void submit(value_t value);
    void reject_received(reject_message const& msg, acceptor_id_t const& acceptor_id);
    get_status_t get(slot_n num, value_t& value, slot_profile_t& profile);

    // Does not replace elements, just accumulates them.
    // When promise answers majority reached, re-submits not commited values from
    // acceptros - an invariant of the protocol.
    void promise_received(promise_message const& msg, acceptor_id_t const& id);

    // wrapper function for unit tests
    bool learn_received(const learn_message& msg, const acceptor_id_t& acceptor_id)
    {
        return learn_received(
            msg.accepted_value.slot, msg.ballot, msg.accepted_value.value, acceptor_id);
    }

    bool learn_received(
        slot_n value_slot,
        ballot_t value_ballot,
        value_t value,
        const acceptor_id_t& acceptor_id);

    void timeout(slot_n n);
    void prepare_timeout();
    void master_announce_timeout();

private:
    void cleanup_frame();
    void drop_value(value_t value);
    bool slot_is_free(slot_t& slot);
    bool insert_proposal(value_t value);
    void propose_one(slot_t& slot);
    // will be called after prepare phase complete or in submit if already prepared
    void propose_all_waiting_values();
    void reset_prepare(ballot_t minimal_ballot);
    void increase_ballot(ballot_t minimal_ballot);
    void start_prepare(ballot_t minimal_ballot);
    void complete_prepare();
    void pull_accumulated_values();
    void cancel_all_async_operations();
    void drop_proposals();
    void start_master_announce();
    void stop_master_announce();
    void cancel_propose_operations();
    bool is_committed(slot_n n);
    bool promised(slot_n slot);

    bool shift_delivery_frame();
    void slot_insert(slot_t& slot, slot_n n, value_t value);
    void slot_accumulate(
        slot_t& slot,
        slot_n n,
        ballot_t ballot,
        value_t value,
        const acceptor_id_t& acceptor_id);
    bool slot_learn(
        slot_n value_slot,
        ballot_t value_ballot,
        value_t value,
        const acceptor_id_t& acceptor_id);
    void slot_propose(slot_t& slot);
    void slot_cancel_propose(slot_t& slot);

    frame_t& frame_;
    stats_t& stats_;
    bool active_;

    // node has successfully completed 'prepare' state (has requested an
    // actual ballot number from acceptors and has received acks)
    bool prepared;
    // state data for the prepare phase
    prepare_state_t prepare_state;

public: // XXX public for unit tests; TODO fix!
    settings_t settings;
    ballot_t ballot; // the current master's working ballot

private:
    slot_n window_end;
    iid_t max_promised; // max promised slot
    // master announces
    timers::timer_ptr announce_timer;
};

}

#undef L_STREAM
#undef DL
#undef MAX_PREPARE_VALUES_SIZE
#undef LOG_IGNORED
