#pragma once

#include <multipaxos/types.h>
#include <multipaxos/paxos_config.h>
#include <multipaxos/settings.h>
#include <multipaxos/frame.h>
#include <multipaxos/slots.h>
#include <multipaxos/messages.h>
#include <multipaxos/ntimer.h>
#include <multipaxos/timers_queue.h>
#include <multipaxos/stats.h>

#include "logger.h"
#include "common_methods.h"

#define LOG_ENABLED settings.log_func
#define L_STREAM                                                                                   \
    if (LOG_ENABLED) logger_t(settings.log_func) << "slave_log "
#define DEBUG_LOG_ENABLED settings.debug_log_func
#define L_DEBUG                                                                                    \
    if (DEBUG_LOG_ENABLED) logger_t(settings.debug_log_func) << "slave_log "

#define FRAME_TO_STREAM                                                                            \
    " read=" << frame_.read_zone_begin() << " write=" << frame_.write_zone_begin()                 \
             << " sync_top=" << sync_top

namespace multipaxos {

class slave_impl
{
public:
    slave_impl(frame_t& frame, const settings_t& settings, timers::queue_ptr timers_queue)
        : frame_(frame), settings(settings), active_(false)
    {
        if (timers_queue) sync_timer = timers_queue->create_timer();
        last_sync_size = 0;
        sync_top = -1;
    }

    bool is_active() const
    {
        return active_;
    }

    void activate()
    {
        assert(!active_);
        active_ = true;
        start_background_synchronization();
    }

    void deactivate()
    {
        assert(active_);
        active_ = false;
        cancel_all_async_operations();
        cleanup_frame();
    }

    void cleanup_frame()
    {
        ntimer_t profile_timer;
        L_STREAM << "cleanup_frame" << FRAME_TO_STREAM;
        for (slot_n i = frame_.write_zone_begin(); i < frame_.write_zone_end(); ++i)
        {
            slot_t& slot = frame_.get_slot(i);
            if (is_committed(frame_, i)) continue;
            if (slot.is_inited())
            {
                L_STREAM << "j_cleanup num=" << i;
            }
            slot.reset();
        }
        L_STREAM << "cleanup_frame finished time=" << to_string(profile_timer) << FRAME_TO_STREAM;
    }

    bool pick_value(
        slot_n value_slot,
        ballot_t value_ballot,
        value_t value,
        acceptor_id_t const& acceptor_id)
    {
        assert(active_);
        if (!frame_.write_zone_has(value_slot)) return false;
        auto& slot = frame_.get_slot(value_slot);
        if (slot.committed_as(value_slot)) return false;
        assert(!slot.value); // slave should not store values
        if (slot.is_inited_as(value_slot)) assert(slot.get_state() == state_t::learn);
        else
        {
            slot.init(value_slot, -1, state_t::learn, value_t());
        }

        bool majority = slot.add_answer(value_slot, value_ballot, value, acceptor_id);
        if (majority)
        {
            L_STREAM << "j_learn_commit num=" << value_slot;
            slot_t& slot = frame_.get_slot(value_slot);
            slot.value = slot.answers.value;
            slot.answers.reset();
            slot.set_state(state_t::committed);
            slot_n committed_range_end = frame_.write_zone_begin();
            for (; committed_range_end < frame_.write_zone_end(); ++committed_range_end)
            {
                if (!is_committed(frame_, committed_range_end))
                {
                    break;
                }
            }
            bool need_notify = (frame_.write_zone_begin() != committed_range_end);
            frame_.extend_read_zone(committed_range_end);
            // notify
            if (need_notify && settings.deliver_func) settings.deliver_func(frame_.read_zone_end());
        }
        else
        {
            L_DEBUG << "j_learn num=" << value_slot;
        }

        return true;
    }

    void cancel_all_async_operations()
    {
        stop_background_synchronization();
    }

    void start_background_synchronization()
    {
        if (sync_timer && settings.send_sync_message)
        {
            sync_timer->async_wait(
                settings.sync_interval, boost::bind(&slave_impl::synchronization_timeout, this));
        }
    }

    void stop_background_synchronization()
    {
        if (sync_timer)
        {
            sync_timer->cancel();
        }
    }

    std::set<slot_n> get_next_slots_to_sync()
    {
        std::set<slot_n> slots;
        for (iid_t i = frame_.write_zone_begin(); i < sync_top && i < frame_.write_zone_end(); ++i)
        {
            if (!is_committed(frame_, i))
            {
                slots.insert(i);
                if (slots.size() >= settings.sync_max_size)
                {
                    break;
                }
            }
        }
        return slots;
    }

    void synchronization_timeout()
    {
        if (!active_) return;

        auto slots = get_next_slots_to_sync();
        if (slots.size() && settings.send_sync_message)
        {
            sync_request_message msg;
            msg.slots = slots;
            settings.send_sync_message(msg);
        }
        start_background_synchronization();
    }

    void sync_received(sync_response_message const& msg, acceptor_id_t const& acceptor_id)
    {
        assert(active_);

        if (sync_timer)
        {
            sync_timer->cancel();
        }
        if (msg.pvalues.items().size())
        {
            L_STREAM << "sync_received front=" << msg.pvalues.items().front().slot
                     << " back=" << msg.pvalues.items().back().slot
                     << " size=" << msg.pvalues.items().size() << FRAME_TO_STREAM;
        }
        learn_message fake_learn_msg;
        last_sync_size = msg.pvalues.items().size();
        for (auto& item : msg.pvalues.items())
        {
            fake_learn_msg.ballot = item.ballot;
            fake_learn_msg.accepted_value.slot = item.slot;
            fake_learn_msg.accepted_value.value = item.value;
            pick_value(
                fake_learn_msg.accepted_value.slot,
                fake_learn_msg.ballot,
                fake_learn_msg.accepted_value.value,
                acceptor_id);
        }
        start_background_synchronization();
    }

    void master_announce_received(master_announce_message const& msg)
    {
        assert(active_);

        if (frame_.write_zone_begin() + settings.lag_to_report <= msg.write)
        {
            L_STREAM << "master_announce_received lag detected"
                     << " master_read=" << msg.read << " master_write=" << msg.write
                     << FRAME_TO_STREAM;

            if (settings.report_func)
            {
                settings.report_func(report_code::lagging);
            }
        }
        sync_top = msg.write;
    }

    slot_n get_sync_top()
    {
        return sync_top;
    }

private:
    frame_t& frame_;
    settings_t settings;
    bool active_;
    // background sync data
    slot_n sync_top; // max slot which slave can request from acceptors
    std::size_t last_sync_size;
    timers::timer_ptr sync_timer;
};

}

#undef L_STREAM
#undef DL
#undef LOG_ENABLED
#undef DEBUG_LOG_ENABLED
#undef L_DEBUG
#undef FRAME_TO_STREAM
