#include <multipaxos/slots.h>

#define CHECK_INIT assert(is_inited());
#define CHECK_INIT_AS(n) assert(is_inited_as(n));

namespace multipaxos {

namespace detail {
inline char state_char(state_t state)
{
    switch (state)
    {
    case state_t::idle:
        return 'I';
    case state_t::wait:
        return 'W';
    case state_t::propose:
        return 'P';
    case state_t::committed:
        return 'C';
    case state_t::learn:
        return 'L';
    case state_t::accumulate:
        return 'A';
    default:
        return '%';
    };
}

}

slot_t::slot_t()
{
    reset();
}

char slot_t::state_char() const
{
    if (is_inited())
    {
        return detail::state_char(state);
    }
    else
    {
        return '-';
    }
}

void slot_t::init(slot_n n, ballot_t b, state_t s, value_t v)
{
    num = n;
    ballot = b;
    state = s;
    value = v;

    answers.reset();
    if (timer)
    {
        timer->cancel();
    }
    profile_timer.reset();
    commit_time = 0;
}

void slot_t::reset()
{
    num = -1;
    ballot = -1;
    state = state_t::idle;
    value.reset();

    answers.reset();
    if (timer) timer->cancel();
    profile_timer.reset();
    commit_time = 0;
}

void slot_t::reset_to_state(state_t new_state)
{
    CHECK_INIT
    init(num, ballot, new_state, value);
}

void slot_t::set_state(state_t new_state)
{
    CHECK_INIT
    state = new_state;
}

bool slot_t::add_answer(
    slot_n value_slot,
    ballot_t value_ballot,
    value_t value,
    acceptor_id_t const& acceptor_id)
{
    CHECK_INIT_AS(value_slot)
    (void)value_slot;

    if (answers.get_count() == 0)
    {
        answers.insert(acceptor_id);
        answers.value = value;
        answers.ballot = value_ballot;
        return false;
    }

    // too old ballot - ignore
    if (answers.ballot > value_ballot)
    {
        return false;
    }

    // if more recent - reset the slot and replace slot's value
    if (answers.ballot < value_ballot)
    {
        answers.reset();
        answers.insert(acceptor_id);
        answers.value = value;
        answers.ballot = value_ballot;
        return false;
    }

    bool inserted = answers.insert(acceptor_id);
    // ignore duplicates
    if (!inserted) return false;

    // if ballot equals, check value
    // algorithm constraint
    if (buffer_equals(answers.value, value))
    {
        size_t answers_count = answers.get_count();
        if (answers_count >= MAJORITY) return true;
        else
            return false;
    }
    else
    {
        reset();
        return false;
    }
}

void slot_t::reset_answers()
{
    CHECK_INIT
    if (answers.get_count()) answers.reset();
}

void slot_t::pull_answers_value()
{
    CHECK_INIT
    // log
}

void slot_t::cancel_timer()
{
    if (timer) timer->cancel();
}

}
