# -*- coding: utf-8 -*-

import codecs
import pytest
import sys
import yaml
from _pytest.skipping import MarkEvaluator

from lib import logger
from lib import settings
from lib.settings import get_mx_type
from lib.types import MxType, EnvType, MailType
from lib.utils import BASE_DIR


def pytest_addoption(parser):
    parser.addoption("-E", action="store", metavar="ENVTYPE",
                     default=EnvType.testing.name, choices=[envtype.name for envtype in EnvType],
                     help="only run tests matching the ENVTYPE.")
    parser.addoption("-M", action="store", metavar="MAILTYPE",
                     default=MailType.bigmail.name, choices=[mailtype.name for mailtype in MailType],
                     help="only run tests matching the MAILTYPE.")
    parser.addoption("-B", action="store", metavar="MXTYPE", choices=[mxtype.name for mxtype in MxType],
                     default="mxback",
                     help="only run tests matching the MXTYPE.")
    parser.addoption("--host", action="store", metavar="MXHOST", default=None,
                     help="custom host to override config")
    parser.addoption("--port", action="store", metavar="MXPORT", type=int, default=None,
                     help="custom port to override config")


@pytest.fixture(scope="session", autouse=True)
def assert_mxtype_is_set(request):
    # workaround for bug: https://github.com/pytest-dev/pytest/issues/2026
    if request.config.getoption("-B") is None:
        raise NotImplementedError(
            "Running all tests at once is not supported, "
            "consider set '-B' option."
        )


def pytest_runtest_setup(item):
    eval_skipif = MarkEvaluator(item, "skipIf")
    if eval_skipif.istrue():
        pytest.skip(eval_skipif.getexplanation())

    item_env_markers = [item.get_closest_marker(envtype.name).name for envtype in EnvType
                        if item.get_closest_marker(envtype.name)]
    item_mail_markers = [item.get_closest_marker(mailtype.name).name for mailtype in MailType
                         if item.get_closest_marker(mailtype.name)]
    item_mxtype_markers = [item.get_closest_marker(mxtype.name).name for mxtype in MxType
                           if item.get_closest_marker(mxtype.name)]

    required_envtype = item.config.getoption("-E")
    required_mailtype = item.config.getoption("-M")
    required_mxtype = item.config.getoption("-B")

    if not (required_envtype in item_env_markers or not item_env_markers):
        pytest.skip("test requires Env: {}".format(required_envtype))
    if not (required_mailtype in item_mail_markers or not item_mail_markers):
        pytest.skip("test requires Mail: {}".format(required_mailtype))
    if item_mxtype_markers:
        if required_mxtype not in item_mxtype_markers:
            pytest.skip("test requires MxType: {}".format(required_mxtype))


def pytest_runtest_logstart(nodeid, location):
    name = _extract_test_case_name(nodeid)
    logger.set_test_case_name(name)


def pytest_runtest_teardown(*args):
    logger.reset_test_case_name()


def pytest_configure(config):
    """Performs initialization steps for test session.
    """
    mx_type = config.getoption("-B")
    env_type = config.getoption("-E")
    mail_type = config.getoption("-M")
    mx_host = config.getoption("--host")
    mx_port = config.getoption("--port")
    settings.initialize(BASE_DIR, mx_type, env_type, mail_type, mx_host, mx_port)

    logger.setup_logging()

    from lib.smtp_send_tools import patch_smtplib_logging  # cyclic import
    patch_smtplib_logging()

    import lib.users
    lib.users.load_from_path(BASE_DIR)

    # Use utf-8 for pipes and redirection
    if not sys.stdout.isatty():
        sys.stdout = codecs.getwriter("utf8")(sys.stdout)

    from lib.xdb import setup_pwd
    if env_type in ("corp", "qa"):
        setup_pwd()

    _setup_yaml()


@pytest.fixture
def is_yarovaya_enabled():
    # Keep this state actual with respect to nwsmtp configs
    return MxType[get_mx_type()] is MxType.mxbackout


def _setup_yaml():
    def str_as_unicode(self, node):
        # Always return unicode
        return self.construct_scalar(node)

    yaml.Loader.add_constructor("tag:yaml.org,2002:str", str_as_unicode)
    yaml.SafeLoader.add_constructor("tag:yaml.org,2002:str", str_as_unicode)


def _extract_test_case_name(nodeid):
    """
    >>> nodeid = (
    ...     "tests/test_plus_after_login.py"
    ...     "::TestPlusAfterLogin"
    ...     "::()::test_plus_after_login[case0]")
    >>> _extract_test_case_name(nodeid)
    'test_plus_after_login.case0'
    """
    name = nodeid.split('::')[-1]
    if name.endswith("]"):
        name = name.replace("[", ".")
        name = name[:-1]
    return name


@pytest.fixture(scope="function")
def maillist(request):
    from lib.users import get_user
    return get_user(request.param)


@pytest.fixture(scope="function")
def recipient(request):
    from lib.users import get_user
    return get_user(request.param)


@pytest.fixture(scope="function")
def recipients(request):
    from lib.users import get_users
    rcpts = request.param
    return get_users(*rcpts)


@pytest.fixture(scope="function")
def sender(request):
    from lib.users import get_user
    return get_user(request.param)
