# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from collections import namedtuple
from urllib import urlencode
from urlparse import urljoin

import requests

from lxml import etree
from lib import ssh
from lib.cache import memoize
from lib.settings import SETTINGS

BLACKBOX_URL = SETTINGS["blackbox"]["url"]
USE_SSH_PROXY = SETTINGS["blackbox"]["use_ssh_proxy"]

# TODO(mafanasev) Replace BlackBoxUser with UserInfo from lib.users
BlackBoxUser = namedtuple("BlackBoxUser", ["email", "uid"])

_cache = {}


@memoize(_cache)
def get_userinfo(login):
    if USE_SSH_PROXY:
        return _create_user_from_xml(login, _get_userinfo_through_proxy(login))
    return _create_user_from_xml(login, _get_userinfo(login))


def _get_userinfo(login):
    url = _build_userinfo_url(login)
    response = requests.get(url, verify=False)
    return response.content


def _get_userinfo_through_proxy(login):
    url = _build_userinfo_url(login)
    cmd = "curl -s '{url}'".format(url=url)
    ssh_client = ssh.get_sshclient()
    stdout, _ = ssh.exec_command(ssh_client, cmd)
    return stdout.read()


def _build_userinfo_url(login):
    query_args = {
        "method": "userinfo",
        "login": login,
        "sid": "smtp",
        "userip": "localhost",
        "dbfields": "account_info.country.uid",
    }
    endpoint = "blackbox/?{query_string}"
    return urljoin(
        BLACKBOX_URL,
        endpoint.format(query_string=urlencode(query_args))
    )


def _create_user_from_xml(login, xml):
    root = etree.fromstring(xml)
    return BlackBoxUser(email=login, uid=root.xpath("/doc/uid")[0].text)
