# -*- coding: utf-8 -*-

from functools import wraps
from itertools import chain


def _get_key(*args, **kwargs):
    return "\x01".join(str(arg) for arg in chain(args, kwargs.itervalues()))


def memoize(cache):
    """
    >>> calls = 0
    >>> cache = {}
    >>> @memoize(cache)
    ... def foo(a, b):
    ...    global calls
    ...    calls += 1
    ...    return a + b
    >>> foo(1, 2), calls
    (3, 1)
    >>> cache.values()
    [3]
    >>> foo(1, 2), calls
    (3, 1)
    >>> foo(2, 1), calls
    (3, 2)
    >>> cache.values()
    [3, 3]
    """
    def decorator(func):
        @wraps(func)
        def decorated(*args, **kwargs):
            key = _get_key(*args, **kwargs)
            if key in cache:
                return cache[key]

            res = func(*args, **kwargs)

            cache[key] = res
            return res
        return decorated
    return decorator
