# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import time

from hamcrest.core.base_matcher import BaseMatcher

from lib.matchers.wait_matcher import wait_for
from lib.psql import find_mail_message_rows_by_field

NONDELIVERY_TIMEOUT = 20  # sec


class DeliveryMatcher(BaseMatcher):
    def __init__(self, field_name, field_value, expected_count):
        self.field_name = field_name
        self.field_value = field_value
        self.expected_count = expected_count
        self.actual_messages_count = 0

    def _matches(self, sender):
        mail_messages_records = find_mail_message_rows_by_field(sender.uid, self.field_name, self.field_value)
        self.actual_messages_count = len(mail_messages_records)
        return self.actual_messages_count == self.expected_count

    def describe_mismatch(self, user, description):
        msg = (
            "В ящике пользователя {email} найдены письма"
            " с {field_name}={field_value}"
            " в количестве {actual_messages_count}")
        msg = msg.format(
            email=user.email, field_name=self.field_name,
            field_value=self.field_value, actual_messages_count=self.actual_messages_count)
        description.append_text(msg)

    def describe_to(self, description):
        msg = (
            "Письмо с {field_name}={field_value}"
            " должно быть доставлено в количестве {expected_count}!")
        msg = msg.format(
            field_name=self.field_name,
            field_value=self.field_value,
            expected_count=self.expected_count)
        description.append_text(msg)


def has_delivered_messages_with_subject(subject, expected_count):
    return wait_for(DeliveryMatcher("subject", subject, expected_count))


def has_delivered_message_with_subject(subject):
    return has_delivered_messages_with_subject(subject, expected_count=1)


def has_delivered_messages_with_msgid(msgid, expected_count):
    return wait_for(DeliveryMatcher("hdr_message_id", msgid, expected_count))


def has_delivered_message_with_msgid(msgid):
    return has_delivered_messages_with_msgid(msgid, 1)


def has_no_delivered_messages_with_subject(subject):
    time.sleep(NONDELIVERY_TIMEOUT)
    return has_delivered_messages_with_subject(subject, expected_count=0)


def has_no_delivered_messages_with_msgid(msgid):
    time.sleep(NONDELIVERY_TIMEOUT)
    return has_delivered_messages_with_msgid(msgid, expected_count=0)
