# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import email

from hamcrest.core.base_matcher import BaseMatcher

from email.message import Message
from lib.auth_results_parser import parse_authentication_results
from lib.headers import Header


def _check_auth_results_for_dkim(message, domain, status):
    if not isinstance(message, Message):
        message = email.message_from_string(message)
    auth_results = message.get_all(Header.AUTHENTICATION_RESULTS)
    if not auth_results:
        return False
    success = 0
    for result in auth_results:
        attrs = parse_authentication_results(result)
        if attrs is not None and attrs["domain"] == domain:
            if attrs["status"] != status:
                return False
            success += 1
    return success > 0


class DKIMSignMatcher(BaseMatcher):
    def __init__(self, domain, status):
        self.domain = domain
        self.status = status

    def _matches(self, message):
        return _check_auth_results_for_dkim(message, self.domain, self.status)

    def describe_mismatch(self, message, description):
        msg = (
            "В сообщении\n\n"
            "{message}\n\n"
            "не найден заголовок Authentication-Results"
            " с результатом проверки DKIM-подписи '{status}' для домена '{domain}'")
        msg = msg.format(message=message, status=self.status, domain=self.domain)
        description.append_text(msg)

    def describe_to(self, description):
        msg = (
            "Ожидается заголовок Authentication-Results"
            " с результатом проверки DKIM-подписи '{status}' для домена '{domain}'")
        msg = msg.format(status=self.status, domain=self.domain)
        description.append_text(msg)


def has_dkim_status(domain, status):
    return DKIMSignMatcher(domain, status)
