# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from hamcrest.core.base_matcher import BaseMatcher

from lib.psql import find_mail_label_rows_by_name


class LabelsMatcher(BaseMatcher):
    def __init__(self, name, type, should_exist):
        self.name = name
        self.type = type
        self.should_exist = should_exist
        self.mismatch_message = ""

    def _matches(self, mail_box_row):
        # Проверяем присутствие искомой метки в общей таблице меток mail.labels в единичном количестве
        mail_labels_rows = find_mail_label_rows_by_name(mail_box_row.uid, self.name)
        labels_with_needed_type = [row for row in mail_labels_rows if (row.type == self.type)]
        if len(labels_with_needed_type) != 1:
            self.mismatch_message = (
                "Меток с именем {name} и типом {type} "
                "в таблице mail.labels нашли в количестве {c}\n{rows}"
            ).format(name=self.name, type=self.type, c=len(labels_with_needed_type), rows=mail_labels_rows)
            return False
        # Из mail.labels узнаем lid метки и проверяем наличие этого lid в исследуемой mail.box записи
        actual_lid = labels_with_needed_type[0].lid
        if actual_lid not in mail_box_row.lids and self.should_exist:
            self.mismatch_message = (
                "Искомая метка с именем {name} типом {type} и lid={lid} не найдена "
                "в поле lids таблицы mail.box = {lids}"
            ).format(name=self.name, type=self.type, lid=actual_lid, lids=mail_box_row.lids)
            return False
        if actual_lid in mail_box_row.lids and not self.should_exist:
            self.mismatch_message = (
                "Искомая метка с именем {name} типом {type} и lid={lid} найдена "
                "в поле lids таблицы mail.box = {lids}"
            ).format(name=self.name, type=self.type, lid=actual_lid, lids=mail_box_row.lids)
            return False

        return True

    def describe_mismatch(self, item, mismatch_description):
        mismatch_description.append_text(self.mismatch_message)

    def describe_to(self, description):
        description.append_text(
            "Метка с именем name {name} должна быть записана в поле lids таблицы mail.box".format(name=self.name))


def has_label(name, type):
    return LabelsMatcher(name, type, True)


def has_no_label(name, type):
    return LabelsMatcher(name, type, False)


def has_user_label(name):
    return LabelsMatcher(name, "user", True)


def has_no_user_label(name):
    return LabelsMatcher(name, "user", False)
