# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from hamcrest import all_of, equal_to, contains_string, starts_with
from hamcrest.core.helpers.wrap_matcher import wrap_matcher

from lib.matchers.base import BaseModifyMatcher
from lib.settings import get_mx_settings

SMTP_OK = 250
ACCEPTED_MESSAGE = "2.0.0 Ok: queued on {mxhost} as ".format(mxhost=get_mx_settings()["host"])
ACCEPTED_INFECTED_MESSAGE = "2.0.0 [3] Message infected by virus: queued on {mxhost} as ".format(
    mxhost=get_mx_settings()["host"])
ACCEPTED_RECIPIENT = "2.1.5 <{recipient}> recipient ok"

SMTP_REJECT = 554
SPAM_REJECTED_MESSAGE = "5.7.1 [2] Message rejected under suspicion of SPAM; https://ya.cc/1IrBc "
VIRUS_REJECTED_MESSAGE = "5.7.1 [3] Message infected by virus "
STRONG_SPAM_REJECTED_MESSAGE = "5.7.1 [1] Message rejected under suspicion of SPAM; https://ya.cc/1IrBc "

SMTP_REJECT_EMAIL = 550
REJECTED_BY_DMARC_MESSAGE_PATTERN = "5.7.1 Email rejected per DMARC policy for {domain} "
TOO_MANY_HEADERS_MESSAGE = "5.3.4 Too many headers "
NO_SUCH_USER_MESSAGE = "5.7.1 No such user!"
REJECTED_BY_TARGET_ADDRESS_POLICY_MESSAGE = "5.7.1 Policy rejection on the target address"

SMTP_REJECT_AUTHENTICATION = 535
REJECTED_BY_INVALID_USER_OR_PWD_MESSAGE = "5.7.8 Error: authentication failed: Invalid user or password!"
REJECTED_BY_SPAM = "5.7.8 Error: authentication failed: Your message looks like spam. You need to use " \
    "web for sending or prove you are not a robot using the following link http://ya.cc/6lIV"
REJECTED_BY_EULA_PATTERN = \
    "5.7.8 Error: authentication failed: Please accept EULA first. https://mail.yandex.ru/for/{domain}"

SMTP_ACCESS_DENIED = 530
ONLY_FOR_INTERNAL_ADDRESS_MESSAGE = "5.7.2 This maillist is only for internal addresses!"

SMTP_GREYLISTING_REJECT = 451
REJECT_BY_GREYLISTING_MESSAGE = "4.7.1 Sorry, the service is currently unavailable. Please come back later."


def is_smtp_code(item_matcher):
    class IsSMTPCode(BaseModifyMatcher):
        description = "smtp code"
        modify = lambda _, item: item[0]

    return IsSMTPCode(wrap_matcher(item_matcher))


def is_smtp_msg(item_matcher):
    class IsSMTPMessage(BaseModifyMatcher):
        description = "smtp message"
        modify = lambda _, item: item[1]

    return IsSMTPMessage(wrap_matcher(item_matcher))


def equals_smtp_response(expected_code, expected_msg):
    return all_of(is_smtp_code(equal_to(expected_code)), is_smtp_msg(equal_to(expected_msg)))


def matches_smtp_response(expected_code, expected_msg):
    return all_of(is_smtp_code(equal_to(expected_code)), is_smtp_msg(contains_string(expected_msg)))


def is_accept_response():
    return all_of(is_smtp_code(equal_to(SMTP_OK)), is_smtp_msg(contains_string(ACCEPTED_MESSAGE)))


def is_reject_by_spam_response():
    return all_of(is_smtp_code(equal_to(SMTP_REJECT)), is_smtp_msg(contains_string(SPAM_REJECTED_MESSAGE)))


def is_reject_by_strongspam_response():
    return all_of(is_smtp_code(equal_to(SMTP_REJECT)),
                  is_smtp_msg(contains_string(STRONG_SPAM_REJECTED_MESSAGE)))


def is_reject_by_dmarc_response(domain):
    return all_of(is_smtp_code(equal_to(SMTP_REJECT_EMAIL)),
                  is_smtp_msg(contains_string(REJECTED_BY_DMARC_MESSAGE_PATTERN.format(domain=domain))))


def is_accept_virus_response():
    return all_of(is_smtp_code(equal_to(SMTP_OK)),
                  is_smtp_msg(contains_string(ACCEPTED_INFECTED_MESSAGE)))


def is_reject_by_virus_response():
    return all_of(is_smtp_code(equal_to(SMTP_REJECT)),
                  is_smtp_msg(contains_string(VIRUS_REJECTED_MESSAGE)))


def is_reject_by_too_many_headers_response():
    return all_of(is_smtp_code(equal_to(SMTP_REJECT_EMAIL)),
                  is_smtp_msg(contains_string(TOO_MANY_HEADERS_MESSAGE)))


def is_reject_by_no_such_user_response():
    return all_of(is_smtp_code(equal_to(SMTP_REJECT_EMAIL)),
                  is_smtp_msg(equal_to(NO_SUCH_USER_MESSAGE)))


def is_reject_by_target_address_policy_response():
    return all_of(is_smtp_code(equal_to(SMTP_REJECT_EMAIL)),
                  is_smtp_msg(equal_to(REJECTED_BY_TARGET_ADDRESS_POLICY_MESSAGE)))


def is_reject_by_eula_response(domain):
    return all_of(is_smtp_code(equal_to(SMTP_REJECT_AUTHENTICATION)),
                  is_smtp_msg(equal_to(REJECTED_BY_EULA_PATTERN.format(domain=domain))))


def is_reject_by_invalid_user_or_pwd_response():
    return all_of(is_smtp_code(equal_to(SMTP_REJECT_AUTHENTICATION)),
                  is_smtp_msg(equal_to(REJECTED_BY_INVALID_USER_OR_PWD_MESSAGE)))


def is_reject_by_spam():
    return all_of(is_smtp_code(equal_to(SMTP_REJECT_AUTHENTICATION)),
                  is_smtp_msg(equal_to(REJECTED_BY_SPAM)))


def is_accept_recipient(recipient):
    return all_of(is_smtp_code(equal_to(SMTP_OK)),
                  is_smtp_msg(equal_to(ACCEPTED_RECIPIENT.format(recipient=recipient))))


def is_reject_by_sending_from_external_address():
    return all_of(is_smtp_code(equal_to(SMTP_ACCESS_DENIED)),
                  is_smtp_msg(equal_to(ONLY_FOR_INTERNAL_ADDRESS_MESSAGE)))


def is_reject_by_greylisting_response():
    return all_of(is_smtp_code(equal_to(SMTP_GREYLISTING_REJECT)),
                  is_smtp_msg(starts_with(REJECT_BY_GREYLISTING_MESSAGE)))
